package jprofilegrid;

import java.util.List;

import javax.swing.JFrame;
import javax.swing.JOptionPane;

import jprofilegrid.calculations.MultipleSequenceAlignment;
import jprofilegrid.calculations.ResidueFrequencyCount;
import jprofilegrid.calculations.ResidueFrequencyCountToProfileGrid;
import jprofilegrid.calculations.Sequence;
import jprofilegrid.model.AnalysisOptions;
import jprofilegrid.model.ConservationRange;
import jprofilegrid.model.ProfileGrid;
import jprofilegrid.presenter.ProfileGridPresenter;
import jprofilegrid.presenter.SimilarityParametersWorker;
import jprofilegrid.presenter.SimilarityParametersWorker.SimilarityParametersWorkerCallback;
import jprofilegrid.readers.CommandLineParser;
import jprofilegrid.readers.MSAReaderController;
import jprofilegrid.writers.ImageFileWriter;

public class JProfileGrid
{
    private static ProfileGridPresenter profileGridPresenter;

    public static void main(String[] args)
    {
        // If the user fails to provide an input file,
        // display the graphical interface.
        if( args.length == 0 )
            profileGridPresenter = new ProfileGridPresenter();
        else
            try
            {
                final CommandLineParser parser = new CommandLineParser( args );
                final MultipleSequenceAlignment multipleSequenceAlignment = MSAReaderController.readInputFile(parser.getFileType(), parser.getFile());

                final AnalysisOptions analysisOptions = new AnalysisOptions(multipleSequenceAlignment.getSequences().size());

                int color = parser.getColor();
                if(color < ConservationRange.COLOR_SETS.length)
                {
                    ConservationRange conservationRange = new ConservationRange(ConservationRange.COLOR_SETS[color],ConservationRange.DEFAULT_CONSERVATION_RANGES);
                    analysisOptions.conservationRange = conservationRange;
                }

                List<Sequence> sequences = multipleSequenceAlignment.getSequences();
                analysisOptions.alignmentConstants = parser.getAlignmentConstants();
                analysisOptions.highlightSequence = sequences.get(parser.getHighlightSequence());
                analysisOptions.referenceSequence = sequences.get(parser.getReferenceSequence());
                analysisOptions.positionRowStart = parser.getPositionRowStart();
                analysisOptions.showValuesAs = parser.getShowValuesAs();
                analysisOptions.skipGaps = parser.getSkipPositionGaps() == 1;
                analysisOptions.sortAscending = parser.getAscending() == 1;
                analysisOptions.sortType = parser.getSortType();
                analysisOptions.inputFilename = parser.getFile().getName();
                analysisOptions.outputFilename = parser.getFile().getName() + ".xls";
                analysisOptions.windowSize = parser.getWindowSize() == 0 ? 5 : parser.getWindowSize();
                analysisOptions.columnsPerTier = parser.getColumnsPerTier() == 0 ? 100 : parser.getColumnsPerTier();
                analysisOptions.threshold = parser.getThreshold();
                analysisOptions.similarityFraction = parser.getSimilarityFraction() == 0.0 ? 1.00 : parser.getSimilarityFraction();
                analysisOptions.lastColumn = multipleSequenceAlignment.getMinSequenceLength();

                if(analysisOptions.threshold > 0)
                {
                    analysisOptions.similarityParametersEnabled = true;
                    analysisOptions.recalculateSimilarityParameters = true;
                }

                ResidueFrequencyCount.analyze(multipleSequenceAlignment, analysisOptions);

                if(analysisOptions.recalculateSimilarityParameters)
                    new SimilarityParametersWorker(multipleSequenceAlignment, analysisOptions, new SimilarityParametersWorkerCallback()
                    {
                        @Override
                        public void onSimilarityParametersComputed()
                        {
                            generateAndOutputProfileGrid(parser, multipleSequenceAlignment, analysisOptions);
                        }
                    });
                else
                    generateAndOutputProfileGrid(parser, multipleSequenceAlignment, analysisOptions);
            }
            catch( NumberFormatException e)
            {
                System.out.println(e);
            }
    }

    private static void generateAndOutputProfileGrid(CommandLineParser commandLineParser,
                                                     MultipleSequenceAlignment multipleSequenceAlignment,
                                                     AnalysisOptions analysisOptions)
    {
        ProfileGrid profileGrid = new ProfileGrid(multipleSequenceAlignment.getSequences().size());

        ResidueFrequencyCountToProfileGrid.computeProfileGrid(profileGrid, multipleSequenceAlignment.getMultipleSequenceAnalysis());
        outputFile(commandLineParser, analysisOptions, profileGrid);
    }

    private static void outputFile(CommandLineParser parser, AnalysisOptions analysisOptions, ProfileGrid profileGrid)
    {
        if(parser.isOutputImage())
        {
            ImageFileWriter tempWriter = new ImageFileWriter();
            tempWriter.createImage(analysisOptions.inputFilename + ".png", profileGrid);
        }
        else
            profileGrid.outputExcelFile();
    }

    public static void onOutOfMemory()
    {
        if(profileGridPresenter != null)
            JOptionPane.showMessageDialog(new JFrame(), "The alignment file was too large to open given\n" +
                                                              "the amount of memory allocated to the JVM.\n\n" +
                                                               "Please rerun JProfileGrid using \n\njava -jar -Xmx[Memory]\n" +
                                                               "where [Memory] is about eight times the size of the file.\n\n" +
                                                               "For example, to read in a 256 Megabyte alignment,\n" +
                                                               "run the command:\n\n" +
                                                               "java -jar -Xmx2048M", "Out of Memory", JOptionPane.ERROR_MESSAGE);

        System.out.println("Out of Memory Error\n\nThe alignment file was too large to open given\n" +
                                                              "the amount of memory allocated to the JVM.\n\n" +
                                                               "Please rerun JProfileGrid using \n\njava -jar -Xmx[Memory]\n" +
                                                               "where [Memory] is about eight times the size of the file.\n\n" +
                                                               "For example, to read in a 256 Megabyte alignment,\n" +
                                                               "run the command:\n\n" +
                                                               "java -jar -Xmx2048M");

        System.exit(0);
    }
}