package jprofilegrid.calculations;
import java.util.Hashtable;
import java.util.List;

import jprofilegrid.constants.SortedAminoAcidNamesAndSymbols;
import jprofilegrid.model.AnalysisOptions;

public class ResidueFrequencyCount
{
	private SortedAminoAcidNamesAndSymbols sortedAminoAcidNamesAndSymbols;

	private int[][] aminoAcidCounts;
	private static final int NUMBER_OF_THREADS = 16;
	private static int numThreads;
	private static Object monitor = new Object();

	public ResidueFrequencyCount( SortedAminoAcidNamesAndSymbols nSortedAminoAcidNamesAndSymbols, int[][] AACounts)
	{
		sortedAminoAcidNamesAndSymbols = nSortedAminoAcidNamesAndSymbols;
		aminoAcidCounts = AACounts;
	}

	/*
	 * Iterates over the multiple sequence alignment,
	 * counting the number of occurences of each
	 * amino acid in each column. The result
	 * is stored in aminoAcidCounts[i][j], where
	 * i is the (i+1)th position in the sequence
	 * and j is the jth amino acid (in alphabetical
	 * order with the blank character last.
	 */
	public static void analyze(final MultipleSequenceAlignment multipleSequenceAlignment, final AnalysisOptions analysisOptions)
	{
		final SortedAminoAcidNamesAndSymbols sortedAminoAcidNamesAndSymbols = SortedAminoAcidNamesAndSymbols.sort(analysisOptions.sortType,
																												  analysisOptions.sortAscending,
																												  analysisOptions.alignmentConstants);

		// Construct a hashtable that links amino acid symbols to their indices.
		final Hashtable<Character, Integer> aaSymbolIndexHashtable = new Hashtable<Character, Integer>();
		final String[] sortedAminoAcidSymbols = sortedAminoAcidNamesAndSymbols.getSortedAminoAcidSymbols();
		for(int i = 0; i < sortedAminoAcidSymbols.length; i++)
			aaSymbolIndexHashtable.put(sortedAminoAcidSymbols[i].toCharArray()[0], i);

		final List<Sequence> sequences = multipleSequenceAlignment.getSequences();

		final int numSymbols   = sortedAminoAcidSymbols.length;
		final int numSequences = analysisOptions.sequenceIndicesInMSA.size();
		final int seqLen 	   = multipleSequenceAlignment.getMinSequenceLength();

		final int AACounts[][] = new int[seqLen][numSymbols];

		numThreads = NUMBER_OF_THREADS;
		int numColPerThread = seqLen / NUMBER_OF_THREADS;
		int leftOvers = seqLen % NUMBER_OF_THREADS;

		for(int i = 0; i < NUMBER_OF_THREADS; i++)
		{
			int rangeStart = i * numColPerThread;
			int rangeEnd = rangeStart + numColPerThread;

			if(i == NUMBER_OF_THREADS - 1)
				rangeEnd += leftOvers;

			final int start = rangeStart;
			final int end   = rangeEnd;

			Runnable aminoAcidCounter = new Runnable()
			{
				@Override
				public void run()
				{
					int[] aminoAcidCounts;
					for(int j = start; j < end; j++)
					{
						aminoAcidCounts = new int[numSymbols];

						for(int i = 0; i < numSequences; i++)
						{
							Sequence currentSequence = sequences.get(analysisOptions.sequenceIndicesInMSA.get(i));
							char currentAminoAcid = currentSequence.getAminoAcidChar(j);

							Integer index = aaSymbolIndexHashtable.get(currentAminoAcid);
							if(index != null && index != numSymbols - 1)
								aminoAcidCounts[index]++;
							else
								if( currentAminoAcid == '-' || currentAminoAcid == '.')
									aminoAcidCounts[numSymbols - 1]++;
						}

						AACounts[j] = aminoAcidCounts;
					}

					synchronized(monitor)
					{
						numThreads--;
						if(numThreads == 0)
							monitor.notify();
					}
				}
			};

			new Thread(aminoAcidCounter).start();
		}

		synchronized(monitor)
		{
			try
			{
				monitor.wait();
			}
			catch (InterruptedException e)
			{
				e.printStackTrace();
			}
		}

		analysisOptions.sortedData = new ResidueFrequencyCount(sortedAminoAcidNamesAndSymbols, AACounts);
	}

	public int[][] getAminoAcidCounts()
	{
		return( aminoAcidCounts );
	}

	public String[] getSortedAminoAcidNames()
	{
		return sortedAminoAcidNamesAndSymbols.getSortedAminoAcidNames();
	}

	public String[] getSortedAminoAcidSymbols()
	{
		return sortedAminoAcidNamesAndSymbols.getSortedAminoAcidSymbols();
	}

	public String getSortType()
	{
		return sortedAminoAcidNamesAndSymbols.getSortType();
	}
}
