package jprofilegrid.calculations;

import java.util.Vector;

import jprofilegrid.constants.AlignmentConstants;
import jprofilegrid.model.AnalysisOptions;
import jprofilegrid.model.Cell;
import jprofilegrid.model.ConservationRange;
import jprofilegrid.model.Format;
import jprofilegrid.model.Numeric;
import jprofilegrid.model.ProfileGrid;
import jprofilegrid.model.Text;
import jprofilegrid.model.Tier;
import jprofilegrid.model.UnknownSymbol;

public class ResidueFrequencyCountToProfileGrid
{
	private static int showValuesAs, numberOfSequences, numberOfAminoAcids, alignmentLength;
	private static Text referenceSequenceName, highlightSequenceName;
	private static Text[] positionInAlignmentRow, motiffNumberingRow,
		majorAminoAcidRow, referenceSequenceRow, highlightSequenceRow, highlightSequenceArray;
	private static Cell[][] aminoAcidCounts;
	private static String referenceSequenceNameString = "", highlightSequenceNameString = "";
	private static Tier entireAlignment;
	private static int[] motiffEndpointIndices, variableRegionEnpointIndices, numberOfMotiffAndVariableRegions;
	private static Vector<Double> plotSimValues;

	public static void computeProfileGrid(ProfileGrid profileGrid, MultipleSequenceAnalysis multipleSequenceAnalysis)
	{
		AnalysisOptions analysisOptions = profileGrid.getAnalysisOptions();
		ResidueFrequencyCount sortedData = analysisOptions.sortedData;

		Vector<UnknownSymbol> unknownSymbols = null;

		if(multipleSequenceAnalysis != null)
		{
			unknownSymbols = new Vector<UnknownSymbol>();
			plotSimValues  = multipleSequenceAnalysis.getCorrected().get(multipleSequenceAnalysis.getWindowLength());
			motiffEndpointIndices = multipleSequenceAnalysis.getMotiffEndpointIndices();
			variableRegionEnpointIndices = multipleSequenceAnalysis.getVariableEndpointIndices();
			unknownSymbols = multipleSequenceAnalysis.getUnknownSymbols();
		}

		numberOfAminoAcids 	   = sortedData.getSortedAminoAcidNames().length;
		alignmentLength 	   = sortedData.getAminoAcidCounts().length;
		positionInAlignmentRow = new Text[alignmentLength];
		highlightSequenceArray = new Text[alignmentLength];
		motiffNumberingRow     = new Text[alignmentLength];
		majorAminoAcidRow      = new Text[alignmentLength];
		referenceSequenceRow   = new Text[alignmentLength];
		highlightSequenceRow   = new Text[alignmentLength];
		aminoAcidCounts 	   = new Cell[alignmentLength][numberOfAminoAcids];
		int[] positionInAlignmentRowWithoutSkippingGaps = new int[alignmentLength];


		ConservationRange conservationRanges  = analysisOptions.conservationRange;
		AlignmentConstants alignmentConstants = analysisOptions.alignmentConstants;
		String inputFilename				  = analysisOptions.inputFilename;
		double threshold 					  = analysisOptions.threshold;
		boolean frequencyColorsEnabled 		  = analysisOptions.frequencyColorsEnabled;
		int highlightType					  = analysisOptions.highlightType;
		int highlightThresholdType 			  = analysisOptions.highlightThresholdType;
		double highlightThreshold 			  = analysisOptions.highlightThreshold;

		Sequence referenceSequence = analysisOptions.referenceSequence;
		if(referenceSequence != null )
			referenceSequenceNameString = referenceSequence.getName();

		Sequence highlightSequence = analysisOptions.highlightSequence;
		if(highlightSequence != null)
			highlightSequenceNameString = highlightSequence.getName();

		referenceSequenceName = new Text(new Format(Format.RIGHT_ALIGNED, true), referenceSequenceNameString);
		highlightSequenceName = new Text(new Format(Format.CENTER_ALIGNED,Format.BOLD),
				highlightSequenceNameString.equalsIgnoreCase(referenceSequenceNameString) ? "" : highlightSequenceNameString);

		if(!highlightSequenceName.getValue().equalsIgnoreCase("") && highlightType == 0)
			highlightSequenceName.getFormat().setBorderColor(analysisOptions.highlightSequenceColour);

		showValuesAs = analysisOptions.showValuesAs;
		numberOfSequences = analysisOptions.sequenceIndicesInMSA.size();

		String[] aminoAcidSymbols = sortedData.getSortedAminoAcidSymbols();
		for( int i = 0; i < alignmentLength; i++ )
		{
			positionInAlignmentRow[i] = new Text(new Format(Format.CENTER_ALIGNED, Format.BOLD), "");
			highlightSequenceArray[i] = new Text(new Format(Format.CENTER_ALIGNED, Format.BOLD), "");
			motiffNumberingRow[i] = new Text(new Format(Format.CENTER_ALIGNED, Format.BOLD), "");
			majorAminoAcidRow[i] = new Text(new Format(Format.CENTER_ALIGNED, Format.BOLD), "");
			referenceSequenceRow[i] = new Text(new Format(Format.CENTER_ALIGNED, Format.BOLD), "");

			highlightSequenceRow[i] = new Text(new Format(Format.CENTER_ALIGNED, Format.BOLD),
					highlightSequenceNameString.equalsIgnoreCase(referenceSequenceNameString)  ||
						highlightSequence == null ? "" :
						highlightSequence.getAminoAcid(i));

			for( int j = 0; j < numberOfAminoAcids; j++ )
			{
				double curData = 0.0;

				if( showValuesAs != 3 )
				{
					curData = sortedData.getAminoAcidCounts()[i][j] / (showValuesAs == 0 ? 1 : (double)numberOfSequences);
					aminoAcidCounts[i][j] = new Numeric(new Format(Format.CENTER_ALIGNED, Format.NOT_BOLD), curData);
				}
				else
					aminoAcidCounts[i][j] = new Text(new Format(Format.CENTER_ALIGNED, Format.NOT_BOLD), "");

				if( showValuesAs == 2)
					if(sortedData.getAminoAcidCounts()[i][j] != 0)
						aminoAcidCounts[i][j] = new Text(new Format(Format.CENTER_ALIGNED, Format.NOT_BOLD), aminoAcidSymbols[j]);
					else
						aminoAcidCounts[i][j] = new Text(new Format(Format.CENTER_ALIGNED, Format.NOT_BOLD), "");
			}
		}

		entireAlignment = new Tier( positionInAlignmentRow, motiffNumberingRow,
				majorAminoAcidRow, referenceSequenceRow, aminoAcidCounts, alignmentConstants, sortedData.getAminoAcidCounts(),
				positionInAlignmentRowWithoutSkippingGaps, highlightSequenceArray);

		entireAlignment.setHighlightSequenceRow(highlightSequenceRow);

		if( multipleSequenceAnalysis != null && analysisOptions.similarityParametersEnabled )
		{
			numberOfMotiffAndVariableRegions = calculateNumberOfMotiffAndVariableRegions();
			calculateMotiffBorderings();
		}

		if( frequencyColorsEnabled )
			calculateConservedCellColors( entireAlignment.getAminoAcidCounts(), conservationRanges,
					numberOfSequences, sortedData);

		calculateMajorRow(entireAlignment, sortedData.getSortedAminoAcidSymbols(), threshold * numberOfSequences, sortedData);
		calculatePositionRow(analysisOptions, entireAlignment.getPositionInAlignmentRow(), positionInAlignmentRowWithoutSkippingGaps);

		if(!highlightSequenceNameString.equals(referenceSequenceNameString))
			calculatePositionRowForHighlightSequence(analysisOptions, entireAlignment.getHighlightSequenceArray(),
				positionInAlignmentRowWithoutSkippingGaps );

		if( multipleSequenceAnalysis != null && analysisOptions.similarityParametersEnabled  )
			calculateMotiffAndVarRegions();

		if( analysisOptions.referenceSequence != null )
		calculateReferenceSequenceRow( analysisOptions.referenceSequence,
				entireAlignment.getReferenceSequenceRow() );

		String[] sortedAminoAcidSymbols = sortedData.getSortedAminoAcidSymbols();
		calculateHighlightSequenceCells(analysisOptions, sortedAminoAcidSymbols, entireAlignment.getAminoAcidCounts(),
									 	sortedData, highlightType, highlightThresholdType, highlightThreshold, showValuesAs);

		profileGrid.setInputFile(inputFilename);
		profileGrid.setEntireAlignment(entireAlignment);
		profileGrid.setSortedAminoAcidNames(sortedData.getSortedAminoAcidNames());
		profileGrid.setSortedAminoAcidSymbols(sortedData.getSortedAminoAcidSymbols());
		profileGrid.setReferenceSequenceName(referenceSequenceName);
		profileGrid.setHighlightSequenceName(highlightSequenceName);
		profileGrid.setMotifEndpointIndices(motiffEndpointIndices);
		profileGrid.setVariableRegionEndpointIndices(variableRegionEnpointIndices);

		if(numberOfMotiffAndVariableRegions != null)
		{
			profileGrid.setNumberOfMotifRegions(numberOfMotiffAndVariableRegions[0]);
			profileGrid.setNumberOfVariableRegions(numberOfMotiffAndVariableRegions[1]);
		}

		profileGrid.setUnknownSymbols(unknownSymbols);
		profileGrid.setPlotSimValues(plotSimValues);
		profileGrid.setNumberOfSequences(numberOfSequences);
	}

	/*
	 * Calculates the number of motiff and variable regions.
	 */
	private static int[] calculateNumberOfMotiffAndVariableRegions()
	{
		int numberOfMotifRegions = motiffEndpointIndices.length / 2;
		int numberOfVariableRegions = variableRegionEnpointIndices.length / 2;
		return( new int[] {numberOfMotifRegions, numberOfVariableRegions} );
	}

	/*
	 * Calculates and updates the format information for those cells which are within
	 * a motiff. (This is the black-bordering calculation).
	 */
	private static void calculateMotiffBorderings()
	{
		Text[] majorAminoAcidRow = entireAlignment.getMajorAminoAcidRow();
		Text[] comparisonSequenceRow = entireAlignment.getReferenceSequenceRow();
		Text[] highlightSequenceRow = entireAlignment.getHighlightSequenceRow();
		Cell[][] aminoAcidCounts = entireAlignment.getAminoAcidCounts();
		Format[] majorAminoAcidRowFormats = new Format[majorAminoAcidRow.length];
		Format[] comparisonSequenceRowFormats = new Format[comparisonSequenceRow.length];
		Format[] highlightSequenceRowFormats = new Format[highlightSequenceRow.length];
		Format[][] aminoAcidCountsFormats = new Format[aminoAcidCounts.length][];


		for( int i = 0; i < entireAlignment.getMajorAminoAcidRow().length; i++ )
			majorAminoAcidRowFormats[i] = entireAlignment.getMajorAminoAcidRow()[i].getFormat();
		for( int i = 0; i < entireAlignment.getReferenceSequenceRow().length; i++ )
			comparisonSequenceRowFormats[i] = entireAlignment.getReferenceSequenceRow()[i].getFormat();
		for( int i = 0; i < entireAlignment.getHighlightSequenceRow().length; i++)
			highlightSequenceRowFormats[i] = entireAlignment.getHighlightSequenceRow()[i].getFormat();
		for( int i = 0; i < entireAlignment.getAminoAcidCounts().length; i++ )
		{
			aminoAcidCountsFormats[i] = new Format[aminoAcidCounts[i].length];
			for( int j = 0; j < entireAlignment.getAminoAcidCounts()[i].length; j++ )
				aminoAcidCountsFormats[i][j] = entireAlignment.getAminoAcidCounts()[i][j].getFormat();
		}

		boolean leftEndpoint = true;
		boolean inMotiff = false;
		int indexIntoMotiffEndpoints = 0;
		for( int i = 0; i < aminoAcidCountsFormats.length && indexIntoMotiffEndpoints < motiffEndpointIndices.length; i++ )
		{
			if( i == motiffEndpointIndices[indexIntoMotiffEndpoints] )
			{
				indexIntoMotiffEndpoints++;
				majorAminoAcidRowFormats[i].setTopBorderEnabled(true);
				aminoAcidCountsFormats[i][aminoAcidCountsFormats[i].length - 1].setBottomBorderEnabled(true);

				if( leftEndpoint )
				{
					inMotiff = true;
					majorAminoAcidRowFormats[i].setLeftBorderEnabled(true);
					comparisonSequenceRowFormats[i].setLeftBorderEnabled(true);
					highlightSequenceRowFormats[i].setLeftBorderEnabled(true);

					for( int j = 0; j < aminoAcidCountsFormats[i].length; j++ )
						aminoAcidCountsFormats[i][j].setLeftBorderEnabled(true);
				}
				else
				{
					inMotiff = false;
					majorAminoAcidRowFormats[i].setRightBorderEnabled(true);
					comparisonSequenceRowFormats[i].setRightBorderEnabled(true);
					highlightSequenceRowFormats[i].setRightBorderEnabled(true);

					for( int j = 0; j < aminoAcidCountsFormats[i].length; j++ )
						aminoAcidCountsFormats[i][j].setRightBorderEnabled(true);

				}
				leftEndpoint = !leftEndpoint;
			}

			if( inMotiff )
			{
				majorAminoAcidRowFormats[i].setTopBorderEnabled(true);
				aminoAcidCountsFormats[i][aminoAcidCountsFormats[i].length - 1].setBottomBorderEnabled(true);
			}
		}
	}

	/*
	 * Calculates where variable regions and motiffs starts and ends so that this information
	 * may be displayed in the "Mot" row.
	 */
	private static void calculateMotiffAndVarRegions()
	{
		int currentNum = 0;
		Text[] motifNumberingRow = entireAlignment.getMorphNumberingRow();

		for( int i = 0; i < motiffEndpointIndices.length; i+=2 )
		{
			for( int j = motiffEndpointIndices[i]; j <= motiffEndpointIndices[i+1]; j++ )
				motifNumberingRow[j].setValue("Mot " + currentNum );
			currentNum++;
		}

		currentNum = 0;
		for( int i = 0; i < variableRegionEnpointIndices.length; i+=2 )
		{
			for( int j = variableRegionEnpointIndices[i]; j <= variableRegionEnpointIndices[i+1]; j++ )
				motifNumberingRow[j].setValue("Var " + currentNum );
			currentNum++;
		}
	}

	/*
	 * Colors each data cell in the profile grid according to the threshold values given in conservationRanges.
	 * Note that the first color in the conservationRange array should be white and will not be applied because
	 * this has the effect of removing the grid-lines (bug in JExcelAPI?).
	 */
	private static void calculateConservedCellColors( Cell[][] aminoAcidCounts,
			ConservationRange conservationRanges, double numberOfSequences, ResidueFrequencyCount sortedData)
	{
		for( int i = 0; i < aminoAcidCounts.length; i++ )
			for( int j = 0; j < aminoAcidCounts[i].length; j++ )
			{
				double maxValSet = 0;
				for( int k = 0; k < conservationRanges.getNumberOfRanges(); k++ )
				{
					if( sortedData.getAminoAcidCounts()[i][j] / (double)numberOfSequences
							>= conservationRanges.getThreshold(k)
							&& maxValSet <= conservationRanges.getThreshold(k) )
					{
						aminoAcidCounts[i][j].getFormat().setBackgroundColor(conservationRanges.getColour(k));
						maxValSet = conservationRanges.getThreshold(k);
					}
				}
			}
	}

	/*
	 * Calculates the major row for the alignment. An amino acid is displayed
	 * in the major row if its frequency is above the threshold and it is NOT
	 * a gap character (gap characters are considered to be '*', '-', or '.'.
	 */
	private static void calculateMajorRow( Tier entireAlignment, String[] sortedAminoAcidSymbols, double threshold,
											ResidueFrequencyCount sortedData)
	{
		for( int i = 0; i < entireAlignment.getMajorAminoAcidRow().length; i++ )
		{
			int[] currentColumn = sortedData.getAminoAcidCounts()[i];
			int indexOfMaxiumumOccuringAminoAcid = getIndexOfMaximumOccuringAminoAcidIn( currentColumn );
			if( currentColumn[indexOfMaxiumumOccuringAminoAcid] > threshold )
				if( !sortedAminoAcidSymbols[indexOfMaxiumumOccuringAminoAcid].equalsIgnoreCase("-") )
				entireAlignment.getMajorAminoAcidRow()[i].setValue(
						sortedAminoAcidSymbols[indexOfMaxiumumOccuringAminoAcid]);
		}
	}

	/*
	 * Returns the amino acid with the highest frequency in the given column. If two amino
	 * acids have the same frequency, then the first encountered is returned (in order of
	 * the chosen sort type).
	 */
	private static int getIndexOfMaximumOccuringAminoAcidIn( int[] aminoAcidCountsColumn )
	{
		int index = 0;
		double maxCounts = 0;
		for( int i = 0; i < aminoAcidCountsColumn.length; i++ )
			if( aminoAcidCountsColumn[i] > maxCounts )
			{
				index = i;
				maxCounts = aminoAcidCountsColumn[i];
			}
		return index;
	}

	/*
	 * Calculates the position in sequence row taking into account gap
	 * characters (which aren't counted).
	 */
	private static void calculatePositionRow( AnalysisOptions analysisOptions, Text[] positionInAlignmentRow, int[] positionInAlignmentRowWithoutSkippingGaps )
	{
		int currentPosition = analysisOptions.positionRowStart;
		Sequence referenceSequence = analysisOptions.referenceSequence;
		boolean startedReferenceSequence = false, endedReferenceSequence = false;
		int startNumber = -1;
		int endNumber = positionInAlignmentRow.length - 1;
		int lastCount = 0;

		for(int i = positionInAlignmentRow.length - 1; i >= 0 && !endedReferenceSequence; i--)
		{
			String aminoAcid = referenceSequence.getAminoAcid(i);
			if(!aminoAcid.equalsIgnoreCase("-") && !aminoAcid.equalsIgnoreCase("."))
			{
				endedReferenceSequence = true;
				endNumber = i;
			}
		}

		for( int i = 0; i < positionInAlignmentRow.length; i++ )
		{
			positionInAlignmentRowWithoutSkippingGaps[i] = currentPosition;
			if(analysisOptions.skipGaps)
			{
				String aminoAcid = referenceSequence.getAminoAcid(i);
				if(	!aminoAcid.equalsIgnoreCase("-") && !aminoAcid.equalsIgnoreCase("."))
				{
					startedReferenceSequence = true;
					positionInAlignmentRow[i].setValue( String.valueOf(currentPosition) );
					if(i == endNumber)
						lastCount = currentPosition + 1;
					currentPosition++;
				}



				if(!startedReferenceSequence)
					startNumber = i;
			}
			else
			{
				positionInAlignmentRow[i].setValue( String.valueOf(currentPosition) );
				currentPosition++;
			}
		}


//		int negativeCounter = startNumber * -1; // this makes negativeCounter insensitive to PositionRowStart

		if(analysisOptions.skipGaps)
		{
			int negativeCounter = analysisOptions.positionRowStart - startNumber - 1;
			for( int i = 0; i <= startNumber; i++ )
			{
				positionInAlignmentRow[i].setValue(String.valueOf(negativeCounter));
				negativeCounter++;
			}

			for(int i = endNumber + 1; i < positionInAlignmentRow.length; i++)
			{
				positionInAlignmentRow[i].setValue(String.valueOf(lastCount));
				lastCount++;
			}
		}
	}

	/*
	 * Calculates the position array for the highlight sequence.
	 * This array is used to provide the user with the dynamic number
	 * that appears to the left of the highlight sequence name
	 * at the top of ProfileGrid.
	 */
	private static void calculatePositionRowForHighlightSequence( AnalysisOptions analysisOptions,
			Text[] highlightSequenceArray, int[] positionInAlignmentRowWithoutSkippingGaps )
	{
		int currentPosition = analysisOptions.positionRowStart;
		Sequence highlightSequence = analysisOptions.highlightSequence;

		if( highlightSequence != null )
			for( int i = 0; i < highlightSequenceArray.length; i++ )
			{
				positionInAlignmentRowWithoutSkippingGaps[i] = currentPosition;
				if( analysisOptions.skipGaps )
				{
					if( !highlightSequence.getAminoAcid(i).equalsIgnoreCase("-") )
					{
						highlightSequenceArray[i].setValue( String.valueOf(currentPosition) );
						currentPosition++;
					}
				}
				else
				{
					highlightSequenceArray[i].setValue( String.valueOf(currentPosition) );
					currentPosition++;
				}
			}
	}

	/*
	 * Initializes the comparison sequence row.
	 */
	private static void calculateReferenceSequenceRow( Sequence referenceSequence, Text[] comparisonSequenceRow )
	{
		for( int i = 0; i < comparisonSequenceRow.length; i++ )
			comparisonSequenceRow[i].setValue( referenceSequence.getAminoAcid(i) );
	}

	/*
	 * Highlights the cells in the highlight sequences that differ from
	 * the comparison sequence (highlighting is done via a blue border).
	 */
	private static void calculateHighlightSequenceCells(AnalysisOptions analysisOptions, String[] aminoAcidSymbols, Cell[][] aminoAcidCounts,
														ResidueFrequencyCount sortedData, int highlightType, int highlightThresholdType, double boxThreshold, int integerOrPercentage)
	{
		Sequence referenceSequence = analysisOptions.referenceSequence;
		Sequence highlightSequence = analysisOptions.highlightSequence;

		if(referenceSequence != null && highlightSequence != null )
			for( int i = 0; i < aminoAcidCounts.length; i++ )
				for( int j = 0; j < aminoAcidCounts[i].length; j++ )
				{
					if(highlightType == 0)
					{
						if( !highlightSequence.getAminoAcid(i).equalsIgnoreCase(referenceSequence.getAminoAcid(i)) &&
								highlightSequence.getAminoAcid(i).equalsIgnoreCase( aminoAcidSymbols[j] ) )
						{
							aminoAcidCounts[i][j].getFormat().setBorderColor(analysisOptions.highlightSequenceColour);
							aminoAcidCounts[i][j].getFormat().setLeftBorderEnabled(true);
							aminoAcidCounts[i][j].getFormat().setRightBorderEnabled(true);
							aminoAcidCounts[i][j].getFormat().setTopBorderEnabled(true);
							aminoAcidCounts[i][j].getFormat().setBottomBorderEnabled(true);
						}
					}
					else
					{
						double currentThreshold = (double)sortedData.getAminoAcidCounts()[i][j];
						if(integerOrPercentage == 1)
							currentThreshold /= (double)numberOfSequences;

						boolean passesThreshold = false;

						if(highlightThresholdType == 0)
							passesThreshold = currentThreshold >= boxThreshold;

						if(highlightThresholdType == 1)
							passesThreshold = currentThreshold == boxThreshold;

						if(highlightThresholdType == 2)
							passesThreshold = currentThreshold <= boxThreshold;

						if(currentThreshold > 0 && passesThreshold )
						{
							aminoAcidCounts[i][j].getFormat().setBorderColor(analysisOptions.highlightSequenceColour);
							aminoAcidCounts[i][j].getFormat().setLeftBorderEnabled(true);
							aminoAcidCounts[i][j].getFormat().setRightBorderEnabled(true);
							aminoAcidCounts[i][j].getFormat().setTopBorderEnabled(true);
							aminoAcidCounts[i][j].getFormat().setBottomBorderEnabled(true);
						}
					}
				}
	}
}