package jprofilegrid.presenter;

import java.awt.Component;
import java.awt.Container;
import java.awt.FileDialog;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import javax.swing.JComboBox;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableModel;

import jprofilegrid.calculations.MultipleSequenceAlignment;
import jprofilegrid.calculations.ResidueFrequencyCount;
import jprofilegrid.calculations.ResidueFrequencyCountToProfileGrid;
import jprofilegrid.calculations.Sequence;
import jprofilegrid.model.AnalysisOptions;
import jprofilegrid.model.ConservationRange;
import jprofilegrid.model.ProfileGrid;
import jprofilegrid.presenter.SimilarityParametersWorker.SimilarityParametersWorkerCallback;
import jprofilegrid.readers.MSAReaderController;
import jprofilegrid.view.alignmentwindow.AlignmentWindow;
import jprofilegrid.view.celldescription.CellDescriptionJFrame;
import jprofilegrid.view.frequencycolors.FrequencyColorsJFrame;
import jprofilegrid.view.gridparameters.GrepSequenceJComboBox;
import jprofilegrid.view.gridparameters.GridParametersJPanel;
import jprofilegrid.view.highlight.HighlightJFrame;
import jprofilegrid.view.main.ExportJPanel;
import jprofilegrid.view.main.FiltersJPanel;
import jprofilegrid.view.main.LoadAlignmentJPanel;
import jprofilegrid.view.main.MainJFrame;
import jprofilegrid.view.main.SimilarityParametersJPanel;
import jprofilegrid.view.metadata.MetaDataColumnsJFrame;
import jprofilegrid.view.overview.OverviewJFrame;
import jprofilegrid.view.profilegrid.ProfileGridJFrame;
import jprofilegrid.writers.FASTAFileWriter;
import jprofilegrid.writers.ImageFileWriter;
import jxl.format.Colour;

public class ProfileGridPresenter implements SimilarityParametersWorkerCallback
{
    private static final int MAX_SEQUENCES_TO_DRAW = 600;
    private static final short SINGLE_DISPLAY = 0, PSEUDO_MIRROR = 1, SELECTION = 2;

    private ProfileGridJFrame profileGridJFrame = new ProfileGridJFrame(this);
    private OverviewJFrame profileGridOverviewJFrame, alignmentOverviewJFrame;
    private MetaDataColumnsJFrame metaDataColumnsJFrame;
    private CellDescriptionJFrame cellDescriptionJFrame = new CellDescriptionJFrame(this);
    private FrequencyColorsJFrame freqColor;
    private HighlightJFrame highlightJFrame = new HighlightJFrame(this);
    private AlignmentWindow alignmentWindow = new AlignmentWindow(this);

    private MultipleSequenceAlignment multipleSequenceAlignment;

    private ProfileGrid topProfileGrid, bottomProfileGrid;
    private MainJFrame mainJFrame = new MainJFrame(this);

    public void openHighlightJFrame()
    {
        int x = mainJFrame.getX(),
            y = mainJFrame.getY(),
            w = mainJFrame.getWidth(),
            h = mainJFrame.getHeight();

        highlightJFrame.setLocation(x + w + 1, y + h / 2);
        onHighlightTypeChanged();
        highlightJFrame.setVisible(true);
    }

    public void openIdentifySequencesDialog()
    {
        int x = mainJFrame.getX(),
            y = mainJFrame.getY(),
            w = mainJFrame.getWidth(),
            h = mainJFrame.getHeight();

        cellDescriptionJFrame.setLocation(x + w + 1, y + h / 4);
        cellDescriptionJFrame.setVisible(true);
        cellDescriptionJFrame.update(cellDescriptionJFrame.getGraphics());
    }

    public void browseForMSAFile()
    {
        FileDialog fileDialog = new FileDialog(mainJFrame, "Load Alignment", FileDialog.LOAD );
        fileDialog.setVisible(true);
        if(fileDialog.getFile() != null)
        {
            String msaFilename = fileDialog.getDirectory() + fileDialog.getFile();
            LoadAlignmentJPanel loadAlignmentPanel = mainJFrame.getLoadAlignmentPanel();
            loadAlignmentPanel.getLoadFileBlank().setText(msaFilename);
            loadAlignmentPanel.getLoadFileButton().setEnabled(true);
            loadAlignmentPanel.setFilenameWithoutPath(fileDialog.getFile());
        }
    }

    public void performLoad()
    {
        LoadAlignmentJPanel loadAlignmentJPanel = mainJFrame.getLoadAlignmentPanel();
        String filename    = loadAlignmentJPanel.getFilename();
        int typeOfMSA 	   = loadAlignmentJPanel.getTypeOfMSA();
        int alignmentType  = loadAlignmentJPanel.getAlignmentType();

        multipleSequenceAlignment = MSAReaderController.loadMultipleSequenceAlignment(typeOfMSA, alignmentType, new File(filename));

        int numberOfSequences = multipleSequenceAlignment.getSequences().size();
        topProfileGrid = new ProfileGrid(numberOfSequences);
        bottomProfileGrid = new ProfileGrid(numberOfSequences);

        SimilarityParametersJPanel similarityParametersJPanel = mainJFrame.getSimilarityParametersPanel();
        GridParametersJPanel gridParametersJPanel			  = mainJFrame.getGridParametersPanel();
        ExportJPanel exportJPanel 							  = mainJFrame.getExportPanel();
        FiltersJPanel filtersJPanel							  = mainJFrame.getFiltersPanel();

        gridParametersJPanel.setSortTypeBlank(multipleSequenceAlignment.getAlignmentConstants().getSortTypes());
        exportJPanel.setOutputFilename(filename  + ".xls");
        exportJPanel.setFinalColumn(multipleSequenceAlignment.getMinSequenceLength());

        int x = mainJFrame.getX(),
            y = mainJFrame.getY(),
            w = mainJFrame.getWidth();

        profileGridJFrame.setLocation(x + w, y);

        if(multipleSequenceAlignment.getSequences().size() > 10000)
            gridParametersJPanel.selectPercentage();

        updateSortSequenceLists();
        updateTopProfileGrid();

        setAllComponentsEnabled(true, gridParametersJPanel);
        setAllComponentsEnabled(true, gridParametersJPanel.getPositionRulerJPanel());
        setAllComponentsEnabled(true, gridParametersJPanel.getFrequencyColorsJPanel());
        setAllComponentsEnabled(true, gridParametersJPanel.getSortAndDualDisplayJPanel());
        setAllComponentsEnabled(true, gridParametersJPanel.getOverviewTypesJPanel());
        setAllComponentsEnabled(true, similarityParametersJPanel);
        setAllComponentsEnabled(true, exportJPanel.getColumnsSelectionPanel());
        setAllComponentsEnabled(true, exportJPanel);
        setAllComponentsEnabled(true, filtersJPanel);

        profileGridJFrame.setVisible(true);

        mainJFrame.pack();
    }

    public void updateSpecificProfileGrid(ProfileGrid profileGrid)
    {
        // First we must load all of the options from the user interface
        SimilarityParametersJPanel similarityParametersJPanel = mainJFrame.getSimilarityParametersPanel();
        GridParametersJPanel gridParametersJPanel			  = mainJFrame.getGridParametersPanel();
        ExportJPanel exportJPanel 							  = mainJFrame.getExportPanel();
        FiltersJPanel filtersJPanel							  = mainJFrame.getFiltersPanel();

        List<Sequence> sequences = multipleSequenceAlignment.getSequences();

        // Retrieve the analysis options for this profile grid.
        AnalysisOptions analysisOptions = profileGrid.getAnalysisOptions();

        // Get the highlight and reference sequences.
        Sequence highlightSequence = getHighlightSequence();
        Sequence referenceSequence = getReferenceSequence();
        if(highlightSequence != null) analysisOptions.highlightSequence = highlightSequence;
        if(referenceSequence != null) analysisOptions.referenceSequence = referenceSequence;

        // Get the type and thresholds for highlighting (in the case of value-highlighting).
        analysisOptions.highlightType 			 = highlightJFrame.getHighlightType();
        analysisOptions.highlightThresholdType 	 = highlightJFrame.getAboveEqualsOrBelow();
        analysisOptions.highlightThreshold 		 = highlightJFrame.getHighlightThreshold();

        // Get the highlight color (for either sequence or numeric highlighting).
        analysisOptions.highlightSequenceColour = ColorManager.getSortedColours()[getHighlightSequenceColor()];

        // Get the parameters for similarity-plot calculation.
        analysisOptions.similarityParametersEnabled = similarityParametersJPanel.similarityParametersEnabled();
        analysisOptions.similarityFraction			= similarityParametersJPanel.getSimilarityFraction();
        analysisOptions.windowSize					= similarityParametersJPanel.getWindowSize();
        analysisOptions.threshold					= similarityParametersJPanel.getThreshold();

        // Get regular expression filtering information.
        analysisOptions.regExFilteringEnabled = filtersJPanel.getRegExFilteringEnabled();

        // Get the data from the grid parameters panel.
        analysisOptions.metaDataFilteringEnabled = gridParametersJPanel.getMetaDataFilteringEnabled();
        analysisOptions.positionRowStart	   	 = gridParametersJPanel.getPositionStart();
        analysisOptions.sortType 		 	   	 = gridParametersJPanel.getSortType();
        analysisOptions.frequencyColorsEnabled 	 = gridParametersJPanel.frequencyColorsEnabled();
        analysisOptions.skipGaps 			  	 = gridParametersJPanel.skipGapsEnabled();
        analysisOptions.showValuesAs			 = gridParametersJPanel.showValueAs();
        analysisOptions.sortAscending 			 = gridParametersJPanel.isAscendingSort();

        // Get the information from the export panel for file export.
        analysisOptions.columnsPerTier = exportJPanel.getColumnsPerTier();
        analysisOptions.outputFilename = exportJPanel.getOutputFilename();

        // Get the constants specific to this alignment.
        analysisOptions.alignmentConstants = multipleSequenceAlignment.getAlignmentConstants();

        // Load the conservation ranges and colors from the frequency colors frame.
        analysisOptions.conservationRange = getFrequencyColorConservationRanges();

        // Next, if the user has chosen one of the filtering types, apply the filtering.
        analysisOptions.sequenceIndicesInMSA = new ArrayList<Integer>();

        for(int i = 0; i < sequences.size(); i++)
        {
            Sequence sequence = sequences.get(i);

            boolean passedRegexFilters = true;
            if(analysisOptions.regExFilteringEnabled)
                passedRegexFilters = sequence.getSequenceNamePassedRegexFilter() && sequence.getSequencePassedRegexFilter();

            boolean passedMetaDataFilter = true;
            if(analysisOptions.metaDataFilteringEnabled)
                passedMetaDataFilter = sequence.getSequencePassedMetaDataFilter();

            boolean passedSelectionfilter = true;
            if(analysisOptions.selectionFilteringEnabled)
                passedSelectionfilter = sequence.getSequencePassedSelectionFilter();

            if(passedRegexFilters && passedMetaDataFilter && passedSelectionfilter)
                analysisOptions.sequenceIndicesInMSA.add(i);
        }

        if(analysisOptions.recalculateSimilarityParameters)
            new SimilarityParametersWorker(multipleSequenceAlignment, analysisOptions, this);

        if(analysisOptions.recalculateSorting)
            ResidueFrequencyCount.analyze(multipleSequenceAlignment, analysisOptions);

        ResidueFrequencyCountToProfileGrid.computeProfileGrid(profileGrid, multipleSequenceAlignment.getMultipleSequenceAnalysis());

        analysisOptions.recalculateSimilarityParameters = false;
        analysisOptions.recalculateSorting = false;
    }

    public void updateTopProfileGrid()
    {
        updateSpecificProfileGrid(topProfileGrid);

        AnalysisOptions analysisOptions = topProfileGrid.getAnalysisOptions();

        updateSortSequenceLists();

        if(profileGridOverviewJFrame != null)
            profileGridOverviewJFrame.resetDisplay(topProfileGrid);

        if(alignmentOverviewJFrame != null)
            alignmentOverviewJFrame.resetDisplay(topProfileGrid);

        String title = multipleSequenceAlignment.getAlignmentName() + " - " + analysisOptions.sequenceIndicesInMSA.size() + " Sequences";

        profileGridJFrame.resetTopProfileGrid(title, topProfileGrid);

        if(profileGridJFrame.getDualDisplayEnabled())
        {
            updateSpecificProfileGrid(bottomProfileGrid);
            profileGridJFrame.resetBottomProfileGrid(bottomProfileGrid);
        }
    }

    public Sequence getHighlightSequence()
    {
        GrepSequenceJComboBox highlightSequenceBlank = highlightJFrame.getHighlightSequenceBlank();

        return highlightSequenceBlank.getSelectedSequence();
    }

    public Sequence getReferenceSequence()
    {
        GridParametersJPanel gridParametersJPanel = mainJFrame.getGridParametersPanel();
        GrepSequenceJComboBox referenceSequenceBlank = gridParametersJPanel.getReferenceSequenceBlank();

        return referenceSequenceBlank.getSelectedSequence();
    }

    public int getHighlightSequenceColor()
    {
        JComboBox highlightSequenceColorBlank = highlightJFrame.getHighlightSequenceColorBlank();
        return highlightSequenceColorBlank.getSelectedIndex();
    }

    public void onHighlightTypeChanged()
    {
        JRadioButton sequenceRadioButton = highlightJFrame.getSequenceRadioButton(),
                     aboveRadioButton	 = highlightJFrame.getAboveRadioButton(),
                     equalsRadioButton	 = highlightJFrame.getEqualsRadioButton(),
                     belowRadioButton	 = highlightJFrame.getBelowRadioButton();

        JPanel highlightValuesJPanel   = highlightJFrame.getHighlightValuesJPanel(),
               highlightSequenceJPanel = highlightJFrame.getHighlightSequenceJPanel();

        boolean sequenceFilteringEnabled = sequenceRadioButton.isSelected();
        setAllComponentsEnabled(!sequenceFilteringEnabled, highlightValuesJPanel);
        aboveRadioButton.setEnabled(!sequenceFilteringEnabled);
        equalsRadioButton.setEnabled(!sequenceFilteringEnabled);
        belowRadioButton.setEnabled(!sequenceFilteringEnabled);
        setAllComponentsEnabled(sequenceFilteringEnabled, highlightSequenceJPanel);
    }

    public void updateSortSequenceLists()
    {
        // Get the reference sequence blank from the grid parameters panel.
        GridParametersJPanel gridParametersJPanel = mainJFrame.getGridParametersPanel();
        GrepSequenceJComboBox referenceSequenceBlank = gridParametersJPanel.getReferenceSequenceBlank();

        // Get the highlight sequence blank.
        GrepSequenceJComboBox highlightSequenceBlank = highlightJFrame.getHighlightSequenceBlank();

        highlightSequenceBlank.removeActionListener(highlightJFrame);
        referenceSequenceBlank.removeActionListener(gridParametersJPanel);

        int selectedReferenceSequence = referenceSequenceBlank.getSelectedSequenceIndex(),
            selectedHighlightSequence = highlightSequenceBlank.getSelectedSequenceIndex();

        AnalysisOptions analysisOptions = topProfileGrid.getAnalysisOptions();
        List<Integer> sequenceIndicesInMSA = analysisOptions.sequenceIndicesInMSA;
        List<Sequence> sequences = multipleSequenceAlignment.getSequences();

        List<Sequence> includedSequences = new ArrayList<Sequence>();
        for( int i = 0; i < sequenceIndicesInMSA.size(); i++ )
            includedSequences.add(sequences.get(sequenceIndicesInMSA.get(i)));

        referenceSequenceBlank.setSequences(includedSequences);
        highlightSequenceBlank.setSequences(includedSequences);

        boolean sortSequences = gridParametersJPanel.getSortingEnabled();
        highlightSequenceBlank.setSequenceListSorted(sortSequences);
        referenceSequenceBlank.setSequenceListSorted(sortSequences);

        if(selectedReferenceSequence < referenceSequenceBlank.getItemCount())
            referenceSequenceBlank.setSelectedIndex(selectedReferenceSequence);

        if(selectedHighlightSequence < highlightSequenceBlank.getItemCount())
        highlightSequenceBlank.setSelectedIndex(selectedHighlightSequence);

        referenceSequenceBlank.addActionListener(gridParametersJPanel);
        highlightSequenceBlank.addActionListener(highlightJFrame);
    }

    public void setSelectedAminoAcid()
    {
        int row = profileGridJFrame.getTopProfileGridSelectedRow();
        int col = profileGridJFrame.getTopProfileGridSelectedColumn();

        if(row >= 0 && col >= 0)
        {
            AnalysisOptions analysisOptions = topProfileGrid.getAnalysisOptions();

            int startIndex = Integer.parseInt(topProfileGrid.getEntireAlignment().getPositionInAlignmentRow()[0].getValue());

            SelectionFilterResults selectionFilterResults = SequenceFilter.selectionFilter(row, col,
                                           multipleSequenceAlignment,
                                           analysisOptions,
                                           mainJFrame.getGridParametersPanel().getDisplayedResidueIfPresent(),
                                           alignmentWindow.isVisible(), startIndex);

            JTextArea sequenceListJTextArea = cellDescriptionJFrame.getSequenceListJTextArea();
            sequenceListJTextArea.setText(selectionFilterResults.getListOfSequences().toString());
            sequenceListJTextArea.setBackground(cellDescriptionJFrame.getBackground());

            StringBuilder alignment = selectionFilterResults.getAlignment();
            StringBuilder sequenceNames = selectionFilterResults.getSequenceNames();

            if(alignmentWindow.isVisible())
            {
                alignmentWindow.updateSequences(alignment.toString());
                alignmentWindow.updateNames(sequenceNames.toString());
            }

            if(profileGridJFrame.getSelectionModeEnabled())
            {
                analysisOptions = bottomProfileGrid.getAnalysisOptions();

                analysisOptions.recalculateSorting = true;
                analysisOptions.selectionFilteringEnabled = true;

                updateSpecificProfileGrid(bottomProfileGrid);
                profileGridJFrame.resetBottomProfileGrid(bottomProfileGrid);
            }
        }
    }

    public TableModel constructAlignmentOverviewTableModel()
    {
        AnalysisOptions analysisOptions = topProfileGrid.getAnalysisOptions();
        List<Integer> sequenceIndicesInMSA = analysisOptions.sequenceIndicesInMSA;
        List<Sequence> sequences = multipleSequenceAlignment.getSequences();

        int seqCount = sequenceIndicesInMSA.size();

        int rows;
        double stepSize = 1.0;
        if(seqCount < MAX_SEQUENCES_TO_DRAW)
        {
            rows     = seqCount;
        }
        else
        {
            rows     = MAX_SEQUENCES_TO_DRAW;
            stepSize = seqCount / ((double)MAX_SEQUENCES_TO_DRAW);
        }

        int cols = multipleSequenceAlignment.getMinSequenceLength();

        DefaultTableModel defaultTableModel = new DefaultTableModel(rows, cols);

        double nextSeq = 0;
        for(int i = 0; i < rows; i++)
        {
            int index = (int)nextSeq;

            if(nextSeq == seqCount)
                index = (int)nextSeq - 1;

            Sequence s = sequences.get(sequenceIndicesInMSA.get(index));

            for(int j = 0; j < cols; j++)
                defaultTableModel.setValueAt(s.getAminoAcid(j), i, j);

             nextSeq += stepSize;
        }

        return defaultTableModel;
    }

    public void setSimilarityPlotComputeButtonDone()
    {
        mainJFrame.getSimilarityParametersPanel().setSimilarityComputationButtonDone();
    }

    public boolean areSequencesSorted()
    {
        return mainJFrame.getGridParametersPanel().getSortSequenceListsJCheckBox().isSelected();
    }

    public HighlightJFrame getHighlightJFrame()
    {
        return highlightJFrame;
    }

    public static void setAllComponentsEnabled(boolean enabled, Container container)
    {
        container.setEnabled(enabled);
        for(Component componenet : container.getComponents())
            componenet.setEnabled(enabled);
    }

    public void showFrequencyColorsSelectionJFrame()
    {
        if(freqColor == null)
        {
            freqColor = new FrequencyColorsJFrame(this);
            freqColor.setLocation(mainJFrame.getX() + mainJFrame.getWidth() + 1, mainJFrame.getY() + mainJFrame.getHeight() / 2);
        }

        freqColor.setVisible(true);
    }

    public void showProfileGridOverviewJFrame()
    {
        if( profileGridOverviewJFrame == null)
        {
            profileGridOverviewJFrame = new OverviewJFrame(this, profileGridJFrame.getTitle() + " ProfileGrid Overview");
            profileGridOverviewJFrame.setLocation(profileGridJFrame.getX(), profileGridJFrame.getY() + profileGridJFrame.getHeight());
        }
        profileGridOverviewJFrame.displayProfileGridOverview(topProfileGrid);
        profileGridOverviewJFrame.setVisible(true);
    }

    public void showAlignmentWindow()
    {
        alignmentWindow.setVisible(true);
        setSelectedAminoAcid();
    }

    public void showAlignmentOverviewJFrame()
    {
        if( alignmentOverviewJFrame == null)
        {
            alignmentOverviewJFrame = new OverviewJFrame(this, profileGridJFrame.getTitle() + " Alignment Overview");
            alignmentOverviewJFrame.setLocation(profileGridJFrame.getX(), profileGridJFrame.getY() + profileGridJFrame.getHeight());
        }

        alignmentOverviewJFrame.displayAlignmentOverview(topProfileGrid.getAnalysisOptions().columnsPerTier, topProfileGrid);
        alignmentOverviewJFrame.setVisible(true);
    }

    public void showMetaDataColumnsJFrame()
    {
        if( metaDataColumnsJFrame == null )
            metaDataColumnsJFrame = new MetaDataColumnsJFrame(this);
        metaDataColumnsJFrame.showMetaDataFileSelectionDialog();
    }

    public void outputExcelFile()
    {
        ProfileGrid profileGrid = setOutputData();
        profileGrid.outputExcelFile();
    }

    public void outputFASTAFile()
    {
        ProfileGrid profileGrid = setOutputData();
        FASTAFileWriter.writeFASTAFile(profileGrid.getAnalysisOptions().outputFilename, multipleSequenceAlignment, profileGrid.getAnalysisOptions());
    }

    public void outputImageFile()
    {
        ProfileGrid profileGrid = setOutputData();
        ImageFileWriter imageWriter = new ImageFileWriter();
        imageWriter.createImage(profileGrid.getAnalysisOptions().outputFilename, profileGrid);
    }

    public void outputPyMOLScript()
    {
        ProfileGrid profileGrid = setOutputData();
        profileGrid.outputPyMOLScript();
    }

    private ProfileGrid setOutputData()
    {
        ExportJPanel exportJPanel = mainJFrame.getExportPanel();


        int selectedProfileGrid = exportJPanel.getSelectedProfileGrid();

        ProfileGrid profileGrid = selectedProfileGrid == 0 ? topProfileGrid : bottomProfileGrid;

        AnalysisOptions analysisOptions = profileGrid.getAnalysisOptions();

        String pdbFile = exportJPanel.getPDBFile();
        analysisOptions.pdbFile = pdbFile;

        int columnsPerTier = exportJPanel.getColumnsPerTier();
        if( columnsPerTier > 0 ) analysisOptions.columnsPerTier = columnsPerTier;

        int firstColumn = exportJPanel.getFirstColumn();
        if(firstColumn> 0) analysisOptions.firstColumn = firstColumn;

        int lastColumn = exportJPanel.getLastColumn();
        if( lastColumn > 0 ) analysisOptions.lastColumn = lastColumn;

        analysisOptions.outputFilename = exportJPanel.getOutputFilename();

        return profileGrid;
    }

    public void updateActiveBins()
    {
        JComboBox numberOfBinsJComboBox = freqColor.getNumberOfBinsJComboBox();
        Vector<JTextField> thresholdValueJTextFields = freqColor.getThresholdValueJTextFields();
        Vector<JComboBox> colorValuesJComboBoxes = freqColor.getColorValuesJComboBoxes();

        for(int i = (Integer)numberOfBinsJComboBox.getSelectedItem(); i > 0; i-- )
        {
            thresholdValueJTextFields.get(i - 1).setEnabled(true);
            colorValuesJComboBoxes.get(i - 1).setEnabled(true);
        }

        for(int i = (Integer)numberOfBinsJComboBox.getSelectedItem(); i < thresholdValueJTextFields.size(); i++ )
        {
            thresholdValueJTextFields.get(i).setEnabled(false);
            colorValuesJComboBoxes.get(i).setEnabled(false);
        }

        updateTopProfileGrid();
    }

    private ConservationRange getFrequencyColorConservationRanges()
    {
        if(freqColor == null)
            return ConservationRange.DEFAULT_CONSERVATION_RANGE;

        Vector<JTextField> thresholdValueJTextFields = freqColor.getThresholdValueJTextFields();
        Vector<JComboBox> colorValuesJComboBoxes = freqColor.getColorValuesJComboBoxes();

        ConservationRange cRange = new ConservationRange();

        for( int i = 0; i < thresholdValueJTextFields.size(); i++ )
        {
            try
            {
                JTextField thresholdValueJTextField = thresholdValueJTextFields.get(i);
                if( thresholdValueJTextField.isEnabled() )
                    cRange.addConservationRange(Double.parseDouble(thresholdValueJTextField.getText()),
                            (Colour)colorValuesJComboBoxes.get(i).getSelectedItem());
            }
            catch(NumberFormatException e)
            {
            }
        }

        return cRange;
    }

    public void showAboutDialog()
    {
        mainJFrame.showAboutDialog();
    }

    public void onRegExFiltersChanged(String sequenceName, String sequence)
    {
        SequenceFilter filter = new SequenceFilter();
        filter.regexFilter(multipleSequenceAlignment, sequenceName, sequence);

        // We reset the index of the selected reference and highlight sequences
        // to avoid array out of bounds exceptions.

        GridParametersJPanel gridParametersJPanel = mainJFrame.getGridParametersPanel();

        GrepSequenceJComboBox highlightSequenceBlank = highlightJFrame.getHighlightSequenceBlank();
        GrepSequenceJComboBox referenceSequenceBlank = gridParametersJPanel.getReferenceSequenceBlank();

        highlightSequenceBlank.removeActionListener(highlightJFrame);
        referenceSequenceBlank.removeActionListener(gridParametersJPanel);

        if(referenceSequenceBlank.getItemCount() > 0)
            referenceSequenceBlank.setSelectedIndex(0);

        if(referenceSequenceBlank.getItemCount() > 0)
            highlightSequenceBlank.setSelectedIndex(0);

        highlightSequenceBlank.addActionListener(highlightJFrame);
        referenceSequenceBlank.addActionListener(gridParametersJPanel);

        topProfileGrid.getAnalysisOptions().recalculateSorting = true;
        bottomProfileGrid.getAnalysisOptions().recalculateSorting = true;
        updateTopProfileGrid();
    }

    public void onMetaDataFilterChanged(List<String> matchingSequences)
    {
        SequenceFilter filter = new SequenceFilter();
        filter.nameListFilter(multipleSequenceAlignment, matchingSequences);
        topProfileGrid.getAnalysisOptions().recalculateSorting = true;
        bottomProfileGrid.getAnalysisOptions().recalculateSorting = true;
        updateTopProfileGrid();
    }

    public void recalculateSorting()
    {
        topProfileGrid.getAnalysisOptions().recalculateSorting = true;
        bottomProfileGrid.getAnalysisOptions().recalculateSorting = true;
        updateTopProfileGrid();
    }

    public void computeSimilarityParameters()
    {
        topProfileGrid.getAnalysisOptions().recalculateSimilarityParameters = true;
        updateTopProfileGrid();
    }

    public void onFilterchange()
    {
        topProfileGrid.getAnalysisOptions().recalculateSorting = true;
        bottomProfileGrid.getAnalysisOptions().recalculateSorting = true;
        updateTopProfileGrid();
    }

    @Override
    public void onSimilarityParametersComputed()
    {
        SimilarityParametersJPanel similarityParametersJPanel = mainJFrame.getSimilarityParametersPanel();
        similarityParametersJPanel.enableComputeSimilarityParametersButton();
        updateTopProfileGrid();
    }

    public void setDualDisplayMode(int selectedIndex)
    {
        ExportJPanel exportJPanel = mainJFrame.getExportPanel();

        switch(selectedIndex)
        {
            case SINGLE_DISPLAY: profileGridJFrame.setDualDisplay(false); exportJPanel.setDualDisplayEnabled(false); break;
            case PSEUDO_MIRROR:  profileGridJFrame.setDualDisplay(true);  exportJPanel.setDualDisplayEnabled(true);  break;
            case SELECTION:      profileGridJFrame.setDualDisplay(true);  exportJPanel.setDualDisplayEnabled(true);
        }

        AnalysisOptions analysisOptions = bottomProfileGrid.getAnalysisOptions();
        analysisOptions.recalculateSorting = true;
        updateSpecificProfileGrid(bottomProfileGrid);
        profileGridJFrame.resetBottomProfileGrid(bottomProfileGrid);
    }

    public void showSelectOutputFileDialog()
    {
        FileDialog fileDialog = new FileDialog(mainJFrame, "Choose Output File", FileDialog.SAVE );
        fileDialog.setVisible(true);
        String outputFileName = fileDialog.getDirectory() + fileDialog.getFile();
        if( fileDialog.getFile() != null )
        {
            ExportJPanel exportJPanel = mainJFrame.getExportPanel();
            exportJPanel.getOutputFileBlank().setText(outputFileName);
        }
    }
}
