package jprofilegrid.readers;

import java.io.File;
import java.util.Vector;

import jprofilegrid.constants.AlignmentConstants;
import jprofilegrid.constants.AminoAcidConstants;
import jprofilegrid.constants.DNAConstants;

public class CommandLineParser
{
    private static String[] COMMAND_LINE_OPTIONS = { "w", "t", "ct", "h", "s", "a", "p", "ps", "g", "d", "v", "i", "f", "c", "sf"};
    private double[] COMMAND_LINE_OPTION_VALUES = new double[COMMAND_LINE_OPTIONS.length];
    private String filename;
    private static final int UNKNOWN_FILE_TYPE = -2;

    public CommandLineParser( String[] arguments )
    {
        // Initialize the file type to the unknown value.
        COMMAND_LINE_OPTION_VALUES[12] = UNKNOWN_FILE_TYPE;

        if( arguments.length > 0 )
        {
            if( arguments[0].contains("-") && arguments[0].contains("help"))
                printDetailedUsageMessage();

            filename = arguments[0];
            Vector<String> options = new Vector<String>();
            Vector<String> parameters = new Vector<String>();
            boolean option = true;
            for( int i = 1; i < arguments.length; i++ )
                if( option )
                {
                    if(!arguments[i].substring(0,1).equals("-") && arguments[i].length() > 1)
                        printUsageMessage();
                    else
                        if( arguments[i].substring(1, arguments[i].length()).equalsIgnoreCase("help") )
                            printDetailedUsageMessage();
                        else
                            options.add(arguments[i].substring(1, arguments[i].length()));

                    if( options.get(options.size() - 1).equalsIgnoreCase("a") )
                        parameters.insertElementAt("1", options.size() - 1);
                    else
                        if( options.get(options.size() - 1).equalsIgnoreCase("g") )
                            parameters.insertElementAt("1", options.size() - 1);
                        else
                            if( options.get(options.size() - 1).equalsIgnoreCase("i") )
                                parameters.insertElementAt("1", options.size() - 1);
                            else
                                option = false;
                }
                else
                {
                    parameters.add(arguments[i]);
                    option = true;
                }

            sort( options, parameters );
        }
        else
            printUsageMessage();
    }

    private void sort( Vector<String> options, Vector<String> parameters )
    {
        for( int i = 0; i < options.size(); i++ )
        {
            boolean optionIdentified = false;
            for( int j = 0; j < COMMAND_LINE_OPTIONS.length; j++ )
            {
                if( options.get(i).equalsIgnoreCase(COMMAND_LINE_OPTIONS[j]) )
                {
                    optionIdentified = true;
                    try
                    {
                        COMMAND_LINE_OPTION_VALUES[j] = Double.parseDouble( parameters.get(i) );
                    }
                    catch(NumberFormatException e)
                    {
                        System.out.println("Parameter " +  parameters.get(j) +" is not an integer.");
                        printUsageMessage();
                    }
                }
            }
            if( !optionIdentified )
                printUnidentifiedOptionMessage( options.get(i) );
        }
    }

    private static void printUnidentifiedOptionMessage( String option )
    {
        System.out.println("Unknown option: " + option );
        printUsageMessage();
    }

    private static void printUsageMessage()
    {
        printStandardUsageOutput();
        System.exit(0);
    }

    private static void printDetailedUsageMessage()
    {
        printStandardUsageOutput();
        printExtendedUsageOutput();
        System.exit(0);
    }
    // TODO update version & date
    // TODO update citation
    private static void printStandardUsageOutput()
    {
        Thread.dumpStack();
        System.out.println("\nJProfileGrid v2.0.5; January 17th, 2013");
        System.out.println("http://www.profilegrid.org");
        System.out.println("Cite publication: Alberto I Roca, Aaron C Abajian, David J Vigerust\n" +
                "ProfileGrids solve the large alignment visualization problem: influenza hemeagglutinin example.\n" +
                "F1000Research 2:2 (2013)\n" +
                "Copyright (c) ProfileGrid.org\nThis program comes with ABSOLUTELY NO WARRANTY.\n" +
                "This is free software, and you are welcome to redistribute it\n" +
                "under certain conditions; see http://www.gnu.org/licenses/.");
        System.out.println("Graphical Frontend: java -jar jprofileGrid.jar");
        System.out.println("Console: java -jar jprofilegrid.jar MSAFile [-f FileType] [-w WindowSize] " +
                "[-c ColumnsPerTier] [-t Threshold] [-h HighlightSequence] [-p ReferenceSequence] [-d ProteinOrDNA] [-v ShowValues] [-s SortType] " +
                "[-a ---Ascending Sort---] " +
                "[-g ---SkipPositionGaps---] [-i --ImageOutput--]"
                );
    }

    private static void printExtendedUsageOutput()
    {
        System.out.println("\nFileType: 0: MSF, 1: FASTA, 2: A2M, 3: Stockholm");
        System.out.println("\nSortType Options:");
        System.out.println("0: Alphabetical-Code (Default)");
        System.out.println("1: Alphabetical-Name");
        System.out.println("2: Age");
        System.out.println("3: Flexibility");
        System.out.println("4: Frequency-EcoliK12");
        System.out.println("5: Hydropathy");
        System.out.println("6: Hydrophobicity");
        System.out.println("7: Helix-Propensity");
        System.out.println("8: Mutability-Dayhoff");
        System.out.println("9: Mutability-Grantham");
        System.out.println("10: Surface-Area");
        System.out.println("11: Volume");
        System.out.println("\nProteinOrDNA Options: 0: Protein, 1: DNA");
        System.out.println("\nShowValues Options: 0: Integer, 1: Frequency, 2: None");
    }

    public File getFile()
    {
        return new File(filename);
    }

    public int getWindowSize()
    {
        return (int)COMMAND_LINE_OPTION_VALUES[0];
    }

    public double getThreshold()
    {
        return COMMAND_LINE_OPTION_VALUES[1];
    }

    public int getColumnsPerTier()
    {
        return (int)COMMAND_LINE_OPTION_VALUES[2];
    }

    public int getHighlightSequence()
    {
        return (int)COMMAND_LINE_OPTION_VALUES[3];
    }

    public int getSortType()
    {
        return (int)COMMAND_LINE_OPTION_VALUES[4];
    }

    public double getAscending()
    {
        return COMMAND_LINE_OPTION_VALUES[5];
    }

    public int getReferenceSequence()
    {
        return (int)COMMAND_LINE_OPTION_VALUES[6];
    }

    public int getPositionRowStart()
    {
        return (int)COMMAND_LINE_OPTION_VALUES[7];
    }

    public double getSkipPositionGaps()
    {
        return COMMAND_LINE_OPTION_VALUES[8];
    }

    public AlignmentConstants getAlignmentConstants()
    {
        if((int)COMMAND_LINE_OPTION_VALUES[9] == AlignmentConstants.DNA)
            return new DNAConstants();

        return new AminoAcidConstants();
    }

    public int getShowValuesAs()
    {
        return (int)COMMAND_LINE_OPTION_VALUES[10];
    }
    public boolean isOutputImage()
    {
        return COMMAND_LINE_OPTION_VALUES[11] == 1 ? true : false;
    }
    public int getFileType()
    {
        return (int)COMMAND_LINE_OPTION_VALUES[12];
    }

    public int getColor()
    {
        return (int)COMMAND_LINE_OPTION_VALUES[13];
    }

    public double getSimilarityFraction()
    {
        return COMMAND_LINE_OPTION_VALUES[14];
    }
}
