package jprofilegrid.view.frequencycolors;

import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Menu;
import java.awt.MenuBar;
import java.awt.MenuItem;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.Vector;

import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import jprofilegrid.model.ConservationRange;
import jprofilegrid.presenter.ColorManager;
import jprofilegrid.presenter.ProfileGridPresenter;
import jprofilegrid.view.ColorComboBoxRenderer;
import jprofilegrid.view.OpenWindowsMenu;
import jxl.format.Colour;
import jxl.format.RGB;

public class FrequencyColorsJFrame extends JFrame implements ActionListener, KeyListener
{
	private static final long serialVersionUID = -9206173067242992333L;
	private JComboBox numberOfThresholds = new JComboBox(new Integer[] {1, 2, 3, 4, 5, 6});

	private GridBagLayout mainLayout = new GridBagLayout();
	private GridBagConstraints constraints = new GridBagConstraints();

	//Frequency
	private Vector<JTextField> thresholdValues = new Vector<JTextField>();
	private Vector<JComboBox> colorBoxes = new Vector<JComboBox>();

	private MenuBar menuBar = new MenuBar();
	private Menu fileMenu = new Menu("File");
	private Menu aboutMenu = new Menu("About");
	private MenuItem aboutMenuItem = new MenuItem("JProfileGrid");
	private MenuItem quitMenuItem = new MenuItem("Quit");

	private JLabel colorSetsJLabel = new JLabel("Color Set:");
	private JComboBox colorSetsJComboBox = new JComboBox(ConservationRange.COLOR_SET_NAMES);

	private ProfileGridPresenter profileGridPresenter;

	public FrequencyColorsJFrame(ProfileGridPresenter profileGridPresenter)
	{
		this.profileGridPresenter = profileGridPresenter;
		setDefaultCloseOperation(JFrame.HIDE_ON_CLOSE);
		setTitle("Frequency Colors");

		setLayout(mainLayout);

		constraints.gridx = 0;
		constraints.gridy = 0;

		initializeMenuBar();
		initializeNumberOfThresholdsBlank();
		initializeColorBlanks();

		constraints.gridy++;
		constraints.gridx = 0;
		add(colorSetsJLabel, constraints);
		constraints.gridx++;
		colorSetsJComboBox.addActionListener(this);
		add(colorSetsJComboBox, constraints);

		setResizable(false);

		pack();
	}

	private void initializeMenuBar()
	{
		menuBar.add(fileMenu);
		OpenWindowsMenu openWindowsMenu = new OpenWindowsMenu();
		addWindowListener(openWindowsMenu);
		menuBar.add(openWindowsMenu);
		menuBar.add(aboutMenu);
		quitMenuItem.addActionListener(this);
		aboutMenu.add(aboutMenuItem);
		aboutMenuItem.addActionListener(this);
		fileMenu.add(quitMenuItem);
		setMenuBar((menuBar));
	}

	private void initializeNumberOfThresholdsBlank()
	{
		constraints.gridwidth = 2;
		JPanel thresholdNumPanel = new JPanel(new FlowLayout());
		thresholdNumPanel.add(new JLabel("Number of Bins:"));
		numberOfThresholds.setSelectedIndex(ConservationRange.getDefaultConservationColors().length - 1);
		numberOfThresholds.addActionListener(this);
		thresholdNumPanel.add(numberOfThresholds);
		add(thresholdNumPanel, constraints);
	}

	private void initializeColorBlanks()
	{
		constraints.gridwidth = 1;
		constraints.insets = new Insets(2, 2, 2, 2);
		initializeLayout();
		initializeLabels();

		for( int i = 0; i < ConservationRange.getDefaultConservationColors().length; i++ )
			addRange(ConservationRange.DEFAULT_CONSERVATION_RANGES[i],
					 ConservationRange.getDefaultConservationColors()[i]);

		addKeyListeners();
		addActionListeners();
	}

	private void initializeLayout()
	{
		constraints.gridx = 0;
		constraints.gridy = 1;
		constraints.anchor = GridBagConstraints.CENTER;
	}

	private void initializeLabels()
	{
		add(new JLabel("Threshold:"), constraints);
		constraints.gridx++;
		add(new JLabel("Color:"), constraints);

		constraints.gridx = 0;
		constraints.gridy = 2;
	}

	private void addRange(Double value, Colour colour)
	{
		thresholdValues.add(new JTextField(String.valueOf(value), 5));
		add(thresholdValues.get(thresholdValues.size() - 1), constraints);
		constraints.gridx++;
		colorBoxes.add(new JComboBox(ColorManager.getSortedColours()));
		colorBoxes.get(colorBoxes.size() - 1).setRenderer(new ColorComboBoxRenderer());
		colorBoxes.get(colorBoxes.size() - 1).setSelectedIndex(ColorManager.getColorIndexWithName(colour.getDescription()));
		add(colorBoxes.get(colorBoxes.size() - 1), constraints);
		constraints.gridx = 0;
		constraints.gridy++;
	}

	private void addKeyListeners()
	{
		for( int i = 0; i < thresholdValues.size(); i++ )
			thresholdValues.get(i).addKeyListener(this);
	}

	private void addActionListeners()
	{
		for( int i = 0; i < colorBoxes.size(); i++ )
		{
			RGB currentColor = ColorManager.getSortedColours()[colorBoxes.get(i).getSelectedIndex()].getDefaultRGB();
			colorBoxes.get(i).setForeground(new Color(currentColor.getRed(), currentColor.getGreen(), currentColor.getBlue()));
			colorBoxes.get(i).setSelectedIndex(colorBoxes.get(i).getSelectedIndex());
			colorBoxes.get(i).addActionListener(this);
		}
	}

	public void actionPerformed(ActionEvent e)
	{
		if( e.getSource() == numberOfThresholds )
		{
			profileGridPresenter.updateActiveBins();
			return;
		}

		if( e.getSource() == quitMenuItem )
		{
			System.exit(0);
			return;
		}

		if( e.getSource() == aboutMenuItem )
		{
			profileGridPresenter.showAboutDialog();
			return;
		}

		for( int i = 0; i < colorBoxes.size(); i++ )
			if( e.getSource() == colorBoxes.get(i) )
			{
				RGB currentColor = ColorManager.getSortedColours()[colorBoxes.get(i).getSelectedIndex()].getDefaultRGB();
				colorBoxes.get(i).setForeground(new Color(currentColor.getRed(), currentColor.getGreen(), currentColor.getBlue()));
				profileGridPresenter.updateTopProfileGrid();
				return;
			}

		if(e.getSource() == colorSetsJComboBox)
		{
			int selectedColorSet = colorSetsJComboBox.getSelectedIndex();
			Colour[] colorSet = ConservationRange.COLOR_SETS[selectedColorSet];
			for(int i = 0; i < colorSet.length; i++)
			{
				JComboBox colorBox = colorBoxes.get(i);
				colorBox.removeActionListener(this);
				int colorIndex = ColorManager.getColorIndexWithName(colorSet[i].getDescription());
				colorBox.setSelectedIndex(colorIndex);
				RGB currentColor = ColorManager.getSortedColours()[colorIndex].getDefaultRGB();
				colorBox.setForeground(new Color(currentColor.getRed(), currentColor.getGreen(), currentColor.getBlue()));
				colorBox.addActionListener(this);
			}
			profileGridPresenter.updateTopProfileGrid();
		}
	}

	public void keyPressed(KeyEvent arg0) {}

	public void keyReleased(KeyEvent e)
	{
		profileGridPresenter.updateTopProfileGrid();
	}

	public JComboBox getNumberOfBinsJComboBox()
	{
		return numberOfThresholds;
	}

	public void keyTyped(KeyEvent arg0) {}

	public Vector<JTextField> getThresholdValueJTextFields()
	{
		return thresholdValues;
	}

	public Vector<JComboBox> getColorValuesJComboBoxes()
	{
		return colorBoxes;
	}
}
