package jprofilegrid.view.profilegrid;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.event.AdjustmentEvent;
import java.awt.event.AdjustmentListener;
import java.awt.event.FocusEvent;

import javax.swing.BorderFactory;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.border.Border;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import jprofilegrid.calculations.ResidueFrequencyCount;
import jprofilegrid.constants.AlignmentConstants;
import jprofilegrid.model.AnalysisOptions;
import jprofilegrid.model.Cell;
import jprofilegrid.model.ConservationRange;
import jprofilegrid.model.Format;
import jprofilegrid.model.ProfileGrid;
import jprofilegrid.model.Text;
import jprofilegrid.model.Tier;
import jprofilegrid.presenter.ProfileGridPresenter;
import jxl.format.RGB;


public class ProfileGridJPanel extends JPanel implements AdjustmentListener, ListSelectionListener
{
	private static final long serialVersionUID = 8810051498829113732L;

	private ProfileGridTableModel profileGridTableModel;
	private JTable profileGridJTable;

	private ProfileGridTableModel symbolSideBarTableModel;
	public JTable symbolSideBarsJTable;

	private ProfileGridScrollPane jscrollPane;

	private boolean scrollLockEnabled = false, isInFocus = false;

	private int horizontalScrollValue = 0;

	private Border unselectedBorder;
	private Border selectedBorder;

	private ProfileGrid profileGrid;

	private ProfileGridPresenter profileGridPresenter;
	private ProfileGridJPanel dualDisplay;
	private int previousRow, previousCol;

	public ProfileGridJPanel(ProfileGridPresenter profileGridPresenter)
	{
		this.profileGridPresenter = profileGridPresenter;
		setLayout(new BorderLayout());
		selectedBorder = BorderFactory.createLineBorder(Color.RED);
		unselectedBorder = BorderFactory.createLineBorder(Color.WHITE);
		setOpaque(true);
	}

	public void setDualDisplay(ProfileGridJPanel dualDisplay)
	{
		this.dualDisplay = dualDisplay;
	}

	public void setProfileGrid(ProfileGrid profileGrid)
	{
		this.profileGrid = profileGrid;

		AnalysisOptions analysisOptions = profileGrid.getAnalysisOptions();

		int showValuesAs = analysisOptions.showValuesAs;
		AlignmentConstants alignmentConstants = analysisOptions.alignmentConstants;
		String[] symbolNames = alignmentConstants.getNames();
		int numSymbols = symbolNames.length + 4;

		profileGridTableModel   = new ProfileGridTableModel(numSymbols, profileGrid.getSequenceLength());
		symbolSideBarTableModel = new ProfileGridTableModel(numSymbols, 2);

		profileGridJTable    = new JTable(profileGridTableModel);
		symbolSideBarsJTable = new JTable(symbolSideBarTableModel);

		profileGridJTable.setCellSelectionEnabled(true);
		profileGridJTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		profileGridJTable.getSelectionModel().addListSelectionListener(this);
		profileGridJTable.getColumnModel().getSelectionModel().addListSelectionListener(this);

		profileGridJTable.getTableHeader().setReorderingAllowed(false);
		profileGridJTable.setDragEnabled(false);
		profileGridJTable.setGridColor(java.awt.Color.LIGHT_GRAY);
		profileGridJTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		profileGridJTable.setDefaultRenderer(Cell.class, new ProfileGridTableCellRenderer(showValuesAs));

		symbolSideBarsJTable.getTableHeader().setReorderingAllowed(false);
		symbolSideBarsJTable.setColumnSelectionAllowed(false);
		symbolSideBarsJTable.setRowSelectionAllowed(false);
		symbolSideBarsJTable.setCellSelectionEnabled(false);

		int preferredHeight = symbolSideBarsJTable.getPreferredScrollableViewportSize().height;
		symbolSideBarsJTable.setPreferredScrollableViewportSize(new Dimension(150, preferredHeight));

		int sortType = analysisOptions.sortType;
		String sortTypeName = analysisOptions.alignmentConstants.getSortTypes()[sortType];
		symbolSideBarsJTable.setDefaultRenderer(Text.class, new FixedAAColumnInTable(sortTypeName.equalsIgnoreCase("Taylor"), true));

		horizontalScrollValue = 0;
		if( jscrollPane != null )
		{
			saveHorizontalScrollPosition();
			remove(jscrollPane);
		}

		jscrollPane = new ProfileGridScrollPane(profileGridJTable, analysisOptions.conservationRange);

		jscrollPane.setRowHeaderView(symbolSideBarsJTable);
		add(jscrollPane, BorderLayout.CENTER);

		if(isInFocus)
			jscrollPane.setBorder(selectedBorder);
		else
			jscrollPane.setBorder(unselectedBorder);
		loadHorizontalScrollPosition();

		jscrollPane.getHorizontalScrollBar().addAdjustmentListener(this);

	    resetDisplayedData();

	    setPreferredSize(new Dimension(jscrollPane.getPreferredSize().width + 225, 27*profileGridJTable.getRowHeight()-2));
	}

	public void resetDisplayedData()
	{
		final String[] sortedNames   = profileGrid.getSortedAminoAcidNames();
		final String[] sortedSymbols = profileGrid.getSortedAminoAcidSymbols();
		for( int i = 0; i < symbolSideBarTableModel.getRowCount(); i++ )
			for( int j = 0; j < symbolSideBarTableModel.getColumnCount(); j++ )
			{
				Format format = new Format(Format.CENTER_ALIGNED, Format.BOLD);
				Text text = new Text(format);

				if( j == 0 )
					if( i >= 4)
						text.setValue(sortedNames[i-4]);
					else
						if( i == 0 )
						{
							int selectedRow = profileGridJTable.getSelectedRow();

							String value = "";
							if(selectedRow >= 0)
								value = profileGridTableModel.getValueAt(selectedRow, profileGridJTable.getSelectedRow()).toString();

							text.setValue(value);
						}

				if( j == 1 )
					switch(i)
					{
						case 0: text.setValue("Posn");  break;
						case 1: text.setValue("Major"); break;
						case 2: text = profileGrid.getReferenceSequenceName(); break;
						case 3: text = profileGrid.getHighlightSequenceName(); break;

						default: text.setValue(sortedSymbols[i-4]);
					}

				symbolSideBarTableModel.setValueAt(text, i, j);
			}

		symbolSideBarsJTable.setRowHeight(12);

		AnalysisOptions analysisOptions = profileGrid.getAnalysisOptions();
		int numberOfRows = analysisOptions.alignmentConstants.getNames().length + 4;

		Tier entireAlignment = profileGrid.getEntireAlignment();

		Text[] positionInAlignmentRow = entireAlignment.getPositionInAlignmentRow(),
			   majorAminoAcidRow      = entireAlignment.getMajorAminoAcidRow(),
			   positionSequenceRow    = entireAlignment.getReferenceSequenceRow(),
			   highlightSequenceRow   = entireAlignment.getHighlightSequenceRow();
		Cell[][] aminoAcidCounts      = entireAlignment.getAminoAcidCounts();

		for( int i = 0; i < profileGridTableModel.getColumnCount(); i++ )
		{
			profileGridJTable.getColumnModel().getColumn(i).setPreferredWidth(30);

			for( int j = 0; j < numberOfRows; j++ )
			{
				Cell cell;

				switch(j)
				{
					case 0: cell = positionInAlignmentRow[i]; break;
					case 1: cell = majorAminoAcidRow[i];	  break;
					case 2: cell = positionSequenceRow[i];	  break;
					case 3: cell = highlightSequenceRow[i];   break;

					default: cell = aminoAcidCounts[i][j-4];
				}

				profileGridTableModel.setValueAt(cell, j, i );
			}
		}

		profileGridJTable.setRowHeight(12);

		repaint();
	}

	public void setScrollLock(boolean enableOrDisable)
	{
		scrollLockEnabled = enableOrDisable;
	}

	private class ProfileGridScrollPane extends JScrollPane
	{
		private static final long serialVersionUID = 1L;
		ConservationRange conservationRange;

		public ProfileGridScrollPane(JTable data, ConservationRange nConservationRange)
		{
			super(data);
			conservationRange = nConservationRange;
		}

		public void paintComponent(Graphics g)
		{
			super.paintComponent(g);

			if(g != null)
			{
				g.clearRect(0, symbolSideBarsJTable.getHeight(), symbolSideBarsJTable.getWidth(), getHeight());

				if(conservationRange.getNumberOfRanges() > 0)
				{
					int rectWidth = symbolSideBarsJTable.getWidth() / conservationRange.getNumberOfRanges();
					for( int i = 0; i < conservationRange.getNumberOfRanges(); i++ )
					{
		        		RGB currentColor = conservationRange.getColour(i).getDefaultRGB();
		        		new Color( currentColor.getRed(), currentColor.getGreen(), currentColor.getBlue());
						g.setColor(new Color( currentColor.getRed(), currentColor.getGreen(), currentColor.getBlue()));

						g.fillRect(i *rectWidth, symbolSideBarsJTable.getHeight() + 3, rectWidth, 20);
					}
				}
			}
		}

	}

	public JScrollPane getScrollPanel()
	{
		return jscrollPane;
	}

	@Override
	public void adjustmentValueChanged(AdjustmentEvent e)
	{
		// Assign twice, this is a known bug in Java 1.5
		if( scrollLockEnabled )
		{
			dualDisplay.getScrollPanel().getHorizontalScrollBar().setValue(jscrollPane.getHorizontalScrollBar().getValue());
			dualDisplay.getScrollPanel().getHorizontalScrollBar().setValue(jscrollPane.getHorizontalScrollBar().getValue());
		}
	}

	public void saveHorizontalScrollPosition()
	{
		horizontalScrollValue = jscrollPane.getHorizontalScrollBar().getValue();
	}

	public void loadHorizontalScrollPosition()
	{
		// Assign twice, this is a known bug in Java 1.5
		jscrollPane.getHorizontalScrollBar().setValue(horizontalScrollValue);
		jscrollPane.getHorizontalScrollBar().setValue(horizontalScrollValue);
	}

	public void setIsInFocus(boolean focus)
	{
		isInFocus = focus;
	}

	public Border getUnselectedBorder()
	{
		return unselectedBorder;
	}

	public void focusLost(FocusEvent e)
	{
	}

	@Override
	public void valueChanged(ListSelectionEvent e)
	{
		int col = profileGridJTable.getSelectedColumn();
		int row = profileGridJTable.getSelectedRow() - 4;

		if(col != previousCol || row != previousRow)
		{
			previousCol = col;
			previousRow = row;

			if(row >= 0)
			{
				Tier entireAlignment = profileGrid.getEntireAlignment();
				AnalysisOptions analysisOptions = profileGrid.getAnalysisOptions();
				ResidueFrequencyCount rfc = analysisOptions.sortedData;

				int[][] aaCounts = rfc.getAminoAcidCounts();

				if(col < aaCounts.length && row < aaCounts[row].length)
				{
					int count = aaCounts[col][row];

					String positionInReferenceSequence = entireAlignment.getPositionInAlignmentRow()[col].getValue();
					String positionInHighlightSequence = entireAlignment.getHighlightSequenceArray()[col].getValue();

					for(int i = 0; i < 4; i++)
					{
						Format format = new Format(Format.CENTER_ALIGNED, Format.BOLD);
						Text text = new Text(format);

						switch(i)
						{
							case 0: text.setValue(rfc.getSortedAminoAcidSymbols()[row]); break;
							case 1: text.setValue(Integer.toString(count));  break;
							case 2: text.setValue(positionInReferenceSequence);	  break;
							case 3: text.setValue(positionInHighlightSequence); break;
						}

						symbolSideBarTableModel.setValueAt(text, i, 0);
					}
				}

				profileGridPresenter.setSelectedAminoAcid();

				repaint();
			}
		}
	}

	public int getSelectedRow()
	{
		return profileGridJTable.getSelectedRow() - 4;
	}

	public int getSelectedColumn()
	{
		return profileGridJTable.getSelectedColumn();
	}
}