package jprofilegrid.view.profilegrid;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.text.DecimalFormat;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.border.Border;
import javax.swing.border.CompoundBorder;
import javax.swing.table.DefaultTableCellRenderer;

import jprofilegrid.model.Cell;
import jprofilegrid.writers.excel.ProfileGridFormatToExcelFormatConverter;
import jxl.format.Alignment;
import jxl.format.RGB;
import jxl.write.WritableCellFormat;


public class ProfileGridTableCellRenderer extends DefaultTableCellRenderer
{
	private static final long serialVersionUID = -686000339606310665L;

	Component cell;
	Object value;
	WritableCellFormat cellFormat;
	private static DecimalFormat decimalFormatter = new DecimalFormat();
	public static int BORDER_THICKNESS = 2;
	private int showValuesAs;

	static
	{
		decimalFormatter.setMaximumFractionDigits(2);
		decimalFormatter.setMinimumFractionDigits(2);
		decimalFormatter.setMaximumIntegerDigits(1);
		decimalFormatter.setMinimumIntegerDigits(1);
	}

	public ProfileGridTableCellRenderer(int showValuesAs)
	{
		this.showValuesAs = showValuesAs;
	}

	private void setBold()
	{
		if(((Cell)value).getFormat().isBold())
			cell.setFont(new Font("Times New Roman", Font.BOLD, 12));
		else
			cell.setFont(new Font("Times New Roman", Font.PLAIN, 12));
	}

	private void setBorders()
	{
		if( cellFormat.hasBorders() )
		{
			Border top = null, bottom = null, left = null, right = null;

			for(int i = 0; i < 4; i++)
			{
				RGB rgb = null;

				switch(i)
				{
					case 0: rgb = cellFormat.getBorderColour(jxl.format.Border.TOP).getDefaultRGB();    break;
					case 1: rgb = cellFormat.getBorderColour(jxl.format.Border.BOTTOM).getDefaultRGB(); break;
					case 2: rgb = cellFormat.getBorderColour(jxl.format.Border.LEFT).getDefaultRGB();   break;
					case 3: rgb = cellFormat.getBorderColour(jxl.format.Border.RIGHT).getDefaultRGB();  break;
				}

				int r = rgb.getRed(), g = rgb.getGreen(), b = rgb.getBlue();
				Color c = (r == 255 && g == 255 && b == 255) ? cell.getBackground() : new Color(r, g, b);


				switch(i)
				{
					case 0: top    = BorderFactory.createMatteBorder(BORDER_THICKNESS, 0, 0, 0, c); break;
					case 1: bottom = BorderFactory.createMatteBorder(0, 0, BORDER_THICKNESS, 0, c); break;
					case 2: left   = BorderFactory.createMatteBorder(0, BORDER_THICKNESS, 0, 0, c);   break;
					case 3: right  = BorderFactory.createMatteBorder(0, 0, 0, BORDER_THICKNESS, c);  break;
				}
			}

			CompoundBorder topAndBottom = new CompoundBorder( top, bottom );
			CompoundBorder leftAndRight = new CompoundBorder( left, right );
			CompoundBorder allBorders   = new CompoundBorder( topAndBottom, leftAndRight );

			setBorder(allBorders);
		}
	}

	private void setAlignment()
	{
		if( cellFormat.getAlignment() == Alignment.CENTRE)
			super.setHorizontalAlignment(JLabel.CENTER);

		if( cellFormat.getAlignment() == Alignment.RIGHT)
			setAlignmentY(JLabel.RIGHT);

		if( cellFormat.getAlignment() == Alignment.LEFT)
			setAlignmentY(JLabel.LEFT);
	}

	private void setBackgroundColor()
	{
		RGB rgbBackColor = cellFormat.getBackgroundColour().getDefaultRGB();
		Color backColor = new Color(rgbBackColor.getRed(), rgbBackColor.getGreen(), rgbBackColor.getBlue() );
		cell.setBackground(backColor);
	}

	private void setForegroundColor()
	{
		RGB foreGroundColor = cellFormat.getFont().getColour().getDefaultRGB();
		Color foreColor = new Color(foreGroundColor.getRed(), foreGroundColor.getGreen(), foreGroundColor.getBlue() );
		cell.setForeground(foreColor);
	}

	private void tryToSetTextAsNumber(Component cell, String text, int row)
	{
		setText(text);

		try
		{
			if(row == 0)
			{
				if(Integer.parseInt(text) % 10 == 0)
					cell.setBackground(Color.LIGHT_GRAY);

				return;
			}

			if(showValuesAs == 0)
			{
				int number = (int)Double.parseDouble(text);
				setText(String.valueOf(number));
			}

			if(showValuesAs == 1)
			{
				Double number = Double.parseDouble(text);
				String formattedNumber = decimalFormatter.format(number);
				setText(String.valueOf(formattedNumber));
			}
		}
		catch(NumberFormatException e)
		{
		}
	}

	public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column)
	{
		Cell excelCell = (Cell)value;
		String excelCellString = excelCell.toString();

		cell = super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
		this.value = value;
		cellFormat = ProfileGridFormatToExcelFormatConverter.extractWritableCellFormat(excelCell);

		setBackgroundColor();
		tryToSetTextAsNumber(cell, excelCellString, row );
		// Get rid of zeros for clarity.
		if( row != 0 )
			setText(excelCellString.equalsIgnoreCase("0.0") || excelCellString.equalsIgnoreCase("0") ? "" : getText());
		setBold();
		setBorders();
		setAlignment();
		setForegroundColor();

		if(isSelected)
		{
			int r = getBackground().getRed(), g = getBackground().getGreen(), b = getBackground().getBlue();
			int reductionAmount = 200;
			int red   = r > reductionAmount ? r - reductionAmount : 0;
			int green = g > reductionAmount ? g - reductionAmount : 0;
			int blue  = b > reductionAmount ? b - reductionAmount : 0;
			Color shadedBackground = new Color(red, green, blue);
			cell.setBackground(shadedBackground);
		}

		return cell;
	}

}
