package jprofilegrid.writers;

import jprofilegrid.constants.AminoAcidConstants;
import jprofilegrid.model.AnalysisOptions;
import jprofilegrid.model.ConservationRange;
import jprofilegrid.model.ProfileGrid;
import jprofilegrid.model.Tier;
import jprofilegrid.model.UnknownSymbol;
import jprofilegrid.model.UnknownSymbolSpecies;
import jprofilegrid.writers.excel.DefaultFonts;
import jprofilegrid.writers.excel.ExcelDataBlockWriter;
import jprofilegrid.writers.excel.ExcelNumericCell;
import jprofilegrid.writers.excel.ExcelTextCell;
import jprofilegrid.writers.excel.ProfileGridCellToExcelCellConverter;
import jprofilegrid.writers.excel.ProfileGridFormatToExcelFormatConverter;
import jxl.write.WritableCellFormat;
import jxl.write.WritableWorkbook;
import jxl.write.WriteException;
import jxl.write.WritableSheet;
import jxl.format.Colour;
import java.io.IOException;
import java.text.DecimalFormat;
import java.util.Vector;


public class ExcelFileWriter
{
	private WritableWorkbook workbook;
	private ProfileGrid profileGrid;
	private WritableSheet currentSheet;
	private ExcelTextCell[] aminoAcidConstantsNames;
	private ExcelNumericCell[][] aminoAcidConstants;
	private ExcelTextCell[] aminoAcidSymbols;
	private ExcelTextCell[] aminoAcidNames;
	private ExcelTextCell comparisonSequenceRowName, highlightSequenceRowName;
	private ExcelDataBlockWriter dataWriter;

	/*
	 * These variables are essential to the ExcelFileWriter.
	 * The process for writing a tier is to initialize
	 * the 'tier' member variable and the startCol
	 * and startRow member variables. Calls to the various
	 * writeTier methods references these three variables.
	 */
	private Tier tier;
	private int startCol;
	private int startRow;

	private static int AMINO_ACID_DATA_ROW_OFFSET = 4;
	private static int AMINO_ACID_DATA_COLUMN_OFFSET = 2;

	private static int AMINO_ACID_NAMES_COLUMN_OFFSET = 0;
	private static int AMINO_ACID_SYMBOLS_COLUMN_OFFSET = 1;

	private static ExcelTextCell POSITION_IN_ALIGNMENT = new ExcelTextCell( DefaultFonts.getBoldAndRightAligned(), "Posn" );
	private static int POSITION_IN_ALIGNMENT_ROW_OFFSET = 0;
	private static int POSITION_IN_ALIGNMENT_COLUMN_OFFSET = 1;

	//private static ExcelTextCell MORPH_ROW = new ExcelTextCell( DefaultFonts.getBoldAndRightAligned(), "Morph" );
	//private static int MORPH_ROW_OFFSET = 1;
	//private static int MORPH_COLUMN_OFFSET = 1;

	private static ExcelTextCell MAJOR_ROW = new ExcelTextCell( DefaultFonts.getBoldAndRightAligned(), "Major" );
	private static int MAJOR_ROW_OFFSET = 1;
	private static int MAJOR_COLUMN_OFFSET = 1;

	private static int COMPARISON_SEQUENCE_ROW_OFFSET = 2;
	private static int COMPARISON_SEQUENCE_COLUMN_OFFSET = 1;

	private static int HIGHLIGHT_SEQUENCE_ROW_OFFSET = 3;
	private static int HIGHLIGHT_SEQUENCE_COLUMN_OFFSET = 1;

	private static int ROWS_PER_TIER = 28;

	private static DecimalFormat formatter = new DecimalFormat();
	static
	{
		formatter.setMaximumFractionDigits(2);
		formatter.setMinimumFractionDigits(2);
		formatter.setMaximumIntegerDigits(1);
		formatter.setMinimumIntegerDigits(1);
	}

	/*
	 * A ProfileGridToExcelFileConverter outputs the calculated ProfileGrid data into an Excel workbook. As
	 * such it may only be constructed given a ProfileGrid object
	 * This ensures that calls to the ExcelProfileGridWriter are valid.
	 */
	public ExcelFileWriter( WritableWorkbook nWorkbook, ProfileGrid nProfileGrid )
	{
		workbook = nWorkbook;
		profileGrid = nProfileGrid;
		setAminoAcidSymbols();
		setAminoAcidNames();
		setAminoAcidConstants();
		comparisonSequenceRowName = new ExcelTextCell(DefaultFonts.getBoldAndCentered(),
				profileGrid.getReferenceSequenceName().getValue());

		WritableCellFormat highlightSequenceRowFormat = ProfileGridFormatToExcelFormatConverter.extractWritableCellFormat(
				profileGrid.getHighlightSequenceName());

		highlightSequenceRowName = new ExcelTextCell(
				highlightSequenceRowFormat, profileGrid.getHighlightSequenceName().getValue());
	}

	/*
	 * Performs the task of writing out the entire cell file. Broken into two parts:
	 * 1. Main alignment sheet output and
	 * 2. Variable/Motif sheet output.
	 */
	public void writeExcelFile(int columnsPerTier, int firstColumn, int lastColumn)
	{
		writePortionOfProfileGridInCurrentSheet( columnsPerTier, firstColumn, lastColumn );
		writeColorLegend();
		//writeVariableAndMotifRegionSheets();
		if(profileGrid.getUnknownSymbols() != null)
		{
			writeUnknownAminoAcidSheet();
			writePlotSimSheet();
		}
		writeWorkbook();
		closeWorkbook();
	}

	/*
	 * Writes out the selected portion of the profile grid in the current sheet using the specified
	 * number of columns per tier and first and last columns.
	 */
	private void writePortionOfProfileGridInCurrentSheet(int columnsPerTier, int firstColumn, int lastColumn )
	{
		AnalysisOptions analysisOptions = profileGrid.getAnalysisOptions();
		int sortType = analysisOptions.sortType;
		String sortTypeName = analysisOptions.alignmentConstants.getSortTypes()[sortType];
		String highlightSequenceName = analysisOptions.highlightSequence.getName();

		createSheet(sortTypeName + "-" +  highlightSequenceName);
		int numberOfColumns = (int)Math.ceil((lastColumn - firstColumn + 1) / columnsPerTier);
		for( int i = 0; i < numberOfColumns; i++ )
		{
			tier = profileGrid.getTier( firstColumn + columnsPerTier * i, firstColumn + columnsPerTier * i + columnsPerTier  );
			startCol = 0;
			startRow = ROWS_PER_TIER * i ;
			writeTier();
		}
		tier = profileGrid.getTier( firstColumn + columnsPerTier * numberOfColumns, lastColumn + 1);
		startCol = 0;
		startRow = ROWS_PER_TIER * numberOfColumns;
		writeTier();
		startCol = 0;
		startRow = ROWS_PER_TIER * (numberOfColumns + 1);
	}

	/*
	 * Writes out all of the unidentified amino acids in a new sheet.
	 */
	private void writeUnknownAminoAcidSheet()
	{
		createSheet("Flags");
		startCol = 0;
		startRow = 0;
		writeUnknownAAs();
	}

	/*
	 * Writes out the PlotSim calculation values for the provided
	 * window size.
	 */
	private void writePlotSimSheet()
	{
		AnalysisOptions analysisOptions = profileGrid.getAnalysisOptions();
		String weighted = analysisOptions.weightedString;

		createSheet("PlotSim_WS_" + profileGrid.getAnalysisOptions().windowSize + "_" + weighted);
		startCol = 0;
		startRow = 0;
		ExcelTextCell[] currentRow = new ExcelTextCell[2];

		DecimalFormat decimalFormat = new DecimalFormat();
		decimalFormat.setMaximumFractionDigits(3);
		decimalFormat.setMinimumFractionDigits(3);

		for( int i = 0; i < profileGrid.getPlotSimValues().size(); i++ )
		{
			currentRow[0] = new ExcelTextCell(DefaultFonts.getDefaultFormat(), String.valueOf(i + 1));
			currentRow[1] = new ExcelTextCell(DefaultFonts.getDefaultFormat(),
					String.valueOf(Double.parseDouble(decimalFormat.format(profileGrid.getPlotSimValues().get(i)))));
			dataWriter.writeRow(currentRow, startCol, startRow + i);
		}
	}

	/*
	 * Writes out the unknown amino acids at startCol, startRow in the
	 * currently open sheet.
	 */
	private void writeUnknownAAs()
	{
		Vector<UnknownSymbol> unknownSymbols = profileGrid.getUnknownSymbols();
		for( int i = 0, row = startRow; i < unknownSymbols.size(); i++ )
		{
			UnknownSymbol unknownSymbol = unknownSymbols.get(i);
			for( int j = 0; j < unknownSymbol.getNumberOfSpecies(); j++ )
			{
				int col = startCol + 2;

				UnknownSymbolSpecies unknownSymbolSpecies = unknownSymbol.getUnknownSymbolSpeciesAndLocation(j);

				dataWriter.writeString(new ExcelTextCell(DefaultFonts.getDefaultFormat(), unknownSymbol.getSymbol()), 0, row);
				dataWriter.writeString(new ExcelTextCell(DefaultFonts.getDefaultFormat(), unknownSymbolSpecies.getSpecies()), 1, row);

				for( int k = 0; k < unknownSymbolSpecies.getNumberOfLocations(); k++)
				{
					dataWriter.writeDouble(new ExcelNumericCell(DefaultFonts.getDefaultFormat(), unknownSymbolSpecies.getLocation(k)), col, row);

					col++;
				}

				row++;
			}
		}
	}

	/*
	 * Writes out a single Tier in the given worksheet.
	 */
	private void writeTier()
	{
		writePositionRow();
	//	writeMorphRow();
		writeMajorRow();
		writeComparisonSequenceRow();
		writeHighlightSequenceRow();
		writeAminoAcidNamesAndSymbols();
		writeAminoAcidCounts();
	}

	/*
	 * Writes out the color legend in the current
	 * sheet.
	 */
	private void writeColorLegend()
	{
		ExcelTextCell THRESHOLD = new ExcelTextCell( DefaultFonts.getBoldAndCentered(), "Threshold");
		ExcelTextCell THRESHOLD_COLOR = new ExcelTextCell( DefaultFonts.getBoldAndCentered(), "Color");
		ExcelTextCell SEQUENCES = new ExcelTextCell( DefaultFonts.getBoldAndCentered(), "Sequences:");

		dataWriter.writeString(THRESHOLD, startCol, startRow);
		dataWriter.writeString(THRESHOLD_COLOR, startCol + 1, startRow);

		ConservationRange conservationRange = profileGrid.getAnalysisOptions().conservationRange;
		for( int i = 0; i < conservationRange.getNumberOfRanges(); i++ )
		{
			dataWriter.writeString(new ExcelTextCell(DefaultFonts.getDefaultFormatAndCentered(),
								   String.valueOf(conservationRange.getThreshold(i))),
								   startCol, startRow + 1 + i);
			Colour colour;

			if( conservationRange.getColour(i) != Colour.WHITE )
			{
				if( conservationRange.getColour(i) == Colour.BLACK )
					colour = Colour.AUTOMATIC;
				else
						colour = conservationRange.getColour(i);

				dataWriter.writeString(new ExcelTextCell(DefaultFonts.getDefaultFormatWithBackgroundColourAndCentered(
									 colour),
									   toTitleCase(conservationRange.getColour(i).getDescription())),
									   startCol + 1, startRow + 1 + i);
			}
			else
				dataWriter.writeString(new ExcelTextCell(DefaultFonts.getDefaultFormatAndCentered(),
						   toTitleCase(conservationRange.getColour(i).getDescription())),
						   startCol + 1, startRow + 1 + i);
		}
		dataWriter.writeString(SEQUENCES, startCol, startRow + 1 + conservationRange.getNumberOfRanges());
		dataWriter.writeDouble(new ExcelNumericCell(DefaultFonts.getDefaultFormatAndCentered(), profileGrid.getNumberOfSequences()),
							   startCol + 1, startRow + 1 + conservationRange.getNumberOfRanges());
	}

	private void writePositionRow()
	{
		dataWriter.writeString(POSITION_IN_ALIGNMENT,
							   startCol + POSITION_IN_ALIGNMENT_COLUMN_OFFSET,
							   startRow + POSITION_IN_ALIGNMENT_ROW_OFFSET);
		dataWriter.writeRow(ProfileGridCellToExcelCellConverter.getCells( tier.getPositionInAlignmentRow() ),
							startCol + POSITION_IN_ALIGNMENT_COLUMN_OFFSET + 1,
							startRow + POSITION_IN_ALIGNMENT_ROW_OFFSET);
	}

/*	private void writeMorphRow()
	{
		dataWriter.writeString(MORPH_ROW,
							   startCol + MORPH_COLUMN_OFFSET,
							   startRow + MORPH_ROW_OFFSET);
		dataWriter.writeRow(ProfileGridCellToExcelCellConverter.getTextCells( tier.getMorphNumberingRow() ),
							startCol + MORPH_COLUMN_OFFSET + 1,
							startRow + MORPH_ROW_OFFSET);
	}
*/
	private void writeMajorRow()
	{
		dataWriter.writeString(MAJOR_ROW,
							   startCol + MAJOR_COLUMN_OFFSET,
							   startRow + MAJOR_ROW_OFFSET);
		dataWriter.writeRow(ProfileGridCellToExcelCellConverter.getCells(tier.getMajorAminoAcidRow()),
						    startCol + MAJOR_COLUMN_OFFSET + 1,
						    startRow + MAJOR_ROW_OFFSET);
	}

	private void writeComparisonSequenceRow()
	{
		dataWriter.writeString(comparisonSequenceRowName,
				   startCol + COMPARISON_SEQUENCE_COLUMN_OFFSET,
				   startRow + COMPARISON_SEQUENCE_ROW_OFFSET);
		dataWriter.writeRow(ProfileGridCellToExcelCellConverter.getCells(tier.getReferenceSequenceRow()),
							  startCol + COMPARISON_SEQUENCE_COLUMN_OFFSET + 1,
							  startRow + COMPARISON_SEQUENCE_ROW_OFFSET);
	}

	private void writeHighlightSequenceRow()
	{
		dataWriter.writeString(highlightSequenceRowName,
				   startCol + HIGHLIGHT_SEQUENCE_COLUMN_OFFSET,
				   startRow + HIGHLIGHT_SEQUENCE_ROW_OFFSET);

		dataWriter.writeRow(ProfileGridCellToExcelCellConverter.getCells(tier.getHighlightSequenceRow()),
							  startCol + HIGHLIGHT_SEQUENCE_COLUMN_OFFSET + 1,
							  startRow + HIGHLIGHT_SEQUENCE_ROW_OFFSET);
	}

	private void writeAminoAcidNamesAndSymbols()
	{
		writeAminoAcidNames(startCol + AMINO_ACID_NAMES_COLUMN_OFFSET,
							startRow + AMINO_ACID_DATA_ROW_OFFSET);
		writeAminoAcidSymbols(startCol + AMINO_ACID_SYMBOLS_COLUMN_OFFSET,
							  startRow + AMINO_ACID_DATA_ROW_OFFSET);
		writeAminoAcidSymbols(startCol + AMINO_ACID_SYMBOLS_COLUMN_OFFSET + tier.getAminoAcidCounts().length + 1,
							  startRow + AMINO_ACID_DATA_ROW_OFFSET);
	}

	private void writeAminoAcidCounts()
	{
		dataWriter.writeBlock(ProfileGridCellToExcelCellConverter.getCells(tier.getAminoAcidCounts()),
				  			  startCol + AMINO_ACID_DATA_COLUMN_OFFSET,
				  			  startRow + AMINO_ACID_DATA_ROW_OFFSET );
	}

	/*
	 * Writes the amino acid symbols column at the specified
	 * starting point in the given sheet.
	 */
	private void writeAminoAcidSymbols(int col, int row )
	{
		dataWriter.writeCol( aminoAcidSymbols, col, row );
	}

	/*
	 * Writes the amino acid names column at the specified
	 * starting point in the desired sheet. Assumes the sheet
	 * already exists.
	 */
	private void writeAminoAcidNames(int col, int row )
	{
		dataWriter.writeCol( aminoAcidNames, col, row );
	}

	/*
	 * Sets the AminoAcidConstants Text[][] array and header String[] array
	 * so that the tiers may be created with the data adjacent to it.
	 */
	private void setAminoAcidConstants()
	{
		aminoAcidConstantsNames = new ExcelTextCell[AminoAcidConstants.CONSTANT_NAMES.length];
		for(int i = 0; i < AminoAcidConstants.CONSTANT_NAMES.length; i++)
			aminoAcidConstantsNames[i] = new ExcelTextCell(DefaultFonts.getBoldAndCentered(),
					AminoAcidConstants.CONSTANT_NAMES[i]);

		aminoAcidConstants =
			new ExcelNumericCell[AminoAcidConstants.ALL_CONSTANTS.length][];
		for( int i = 0; i < AminoAcidConstants.ALL_CONSTANTS.length; i++ )
			aminoAcidConstants[i] = new ExcelNumericCell[AminoAcidConstants.ALL_CONSTANTS[i].length];

		for(int i = 0; i < profileGrid.getSortedAminoAcidSymbols().length - 1; i++ )
		{
			AnalysisOptions analysisOptions = profileGrid.getAnalysisOptions();
			double[] oldAminoAcidConstants = analysisOptions.alignmentConstants.getConstantsForSymbol(profileGrid.getSortedAminoAcidSymbols()[i]);
			for(int j = 0; j < oldAminoAcidConstants.length; j++)
				// Because of the way that the amino acids are sorted, it is necessary to transpose the array.
				aminoAcidConstants[j][i] =
					new ExcelNumericCell( DefaultFonts.getDefaultFormat(), oldAminoAcidConstants[j] );
		}
	}

	/*
	 * Sets the aminoAcidSymbols class member via the data in the profileGrid.
	 * This should, in practice, be called only once by the constructor.
	 */
	private void setAminoAcidSymbols()
	{
		String[] oldAminoAcidSymbols = profileGrid.getSortedAminoAcidSymbols();
		ExcelTextCell[] nAminoAcidSymbols = new ExcelTextCell[oldAminoAcidSymbols.length];
		for( int i = 0; i < nAminoAcidSymbols.length; i++ )
			nAminoAcidSymbols[i] = new ExcelTextCell(DefaultFonts.getBoldAndCentered(), oldAminoAcidSymbols[i]);
		aminoAcidSymbols = nAminoAcidSymbols;
	}

	/*
	 * Sets the aminoAcidNames class member via the data in the profileGrid.
	 * This should, in practice, be called only once by the constructor.
	 */
	private void setAminoAcidNames()
	{
		String[] oldAminoAcidNames = profileGrid.getSortedAminoAcidNames();
		ExcelTextCell[] nAminoAcidNames = new ExcelTextCell[oldAminoAcidNames.length];
		for( int i = 0; i < nAminoAcidNames.length; i++ )
			nAminoAcidNames[i] = new ExcelTextCell(DefaultFonts.getBoldAndRightAligned(), oldAminoAcidNames[i]);
		aminoAcidNames = nAminoAcidNames;
	}

	/*
	 * Writes the current buffer to the workbook. In practice this should be called
	 * only once followed by a call to closeWorkbook();
	 */
	public void writeWorkbook()
	{
		try
		{
			workbook.write();
		}
		catch(IOException e)
		{
		}
	}
	/*
	 * Closes the associated workbook. The current instance of ExcelProfileGridWriter should
	 * not be used after calling this function.
	 *
	 */
	public void closeWorkbook()
	{
		try
		{
			workbook.close();
		}
		catch(IOException e)
		{
		}
		catch(WriteException e)
		{
		}
	}

	/*
	 * Create a new sheet at the end of the workbook and open it.
	 */
	private void createSheet(String sheetName)
	{
		currentSheet = workbook.createSheet(sheetName, workbook.getNumberOfSheets());
		dataWriter = new ExcelDataBlockWriter( currentSheet );
	}

	/*
	 * Returns the given String in Title Case.
	 */
	private static String toTitleCase(String string)
	{
		String newString = "";
		boolean cap = true;
		for(int i = 0; i < string.length(); i++ )
		{
			if(cap)
				newString += String.valueOf(string.charAt(i)).toUpperCase();
			else
				newString += string.charAt(i);

			if(newString.substring(newString.length() - 1, newString.length()).equalsIgnoreCase(" "))
				cap = true;
			else
				cap = false;
		}

		return newString;
	}
}
