package jprofilegrid.writers;

import java.awt.AWTException;
import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.Robot;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.text.DecimalFormat;
import java.util.Arrays;
import java.util.List;

import javax.imageio.ImageIO;

import jprofilegrid.model.AnalysisOptions;
import jprofilegrid.model.Cell;
import jprofilegrid.model.Format;
import jprofilegrid.model.Numeric;
import jprofilegrid.model.ProfileGrid;
import jprofilegrid.model.Text;
import jprofilegrid.model.Tier;
import jxl.format.Colour;
import jxl.format.RGB;

/*
 *  Convenience class to create and optionally save to a file a
 *  BufferedImage of an area on the screen. Generally there are
 *  four different scenarios. Create an image of:
 *
 *  a) an entire component
 *  b) a region of the component
 *  c) the entire desktop
 *  d) a region of the desktop
 *
 *  The first two use the Swing paint() method to draw the
 *  component image to the BufferedImage. The latter two use the
 *  AWT Robot to create the BufferedImage.
 *
 *	The createImage(...) methods will also save the image to a file
 *  when the specified file name is not null.
 *
 *  Note: although this class was originally designed to create an image of a
 *  component on the screen it can be used to create an image of components
 *  not displayed on a GUI. For individual components the component is painted
 *  at its preferred size. This can be changed by using the setSize() method
 *  on the component. For a JPanel you need to invoke the setSize() and
 *  doLayout() methods first before attempting to create the image.
 */
/**
 * @author aroca
 *
 */
public class ImageFileWriter
{
	/**
	 * The following constants are in pixels. It is assumed that each
	 * cell in the table will be of a fixed size and the only dynamic
	 * component is the number of cells.
	 */
	private final int BORDER_WIDTH = 2;
	private final int CELL_HEIGHT = 15;// + BORDER_WIDTH;
	// TODO cell width needs to expand depending upon number of sequences. currently ok for <10,000
	private int CELL_WIDTH = 0;

	private final int HEADER_HEIGHT = 4*CELL_HEIGHT;

	/**
	 * The grid offset is the number of pixels on the left where all of
	 * the labels are
	 */
	private final int GRID_OFFSET = 35;

	private List<String> types = Arrays.asList( ImageIO.getWriterFormatNames() );




	/**
	 * Write an jpg representation of this data to the file system at the specified location
	 * @param fileName Fully qualified filename.jpg to be writen to.
	 */
	public void createImage(String fileName, ProfileGrid profileGrid)
	{
		CELL_WIDTH = Math.max(40, 10 * Integer.toString(profileGrid.getNumberOfSequences()).length());
		/**
		 * Calling ProfileGrid component(?)
		 * In a proper OO place, this would be passed in. No Global Data would ever exist
		 */
		AnalysisOptions analysisOptions = profileGrid.getAnalysisOptions();
		String[] names = analysisOptions.alignmentConstants.getNames();
		int heightInRows = names.length;

		Tier entireAlignment = profileGrid.getEntireAlignment();
		Cell[][] aminoAcidEntries = entireAlignment.getAminoAcidCounts();
		int widthInColumns = aminoAcidEntries.length;

		// Figure out the size of the image based on the width and height + border info
		BufferedImage image = new BufferedImage((GRID_OFFSET + (widthInColumns * CELL_WIDTH) + 1), (heightInRows * CELL_HEIGHT + 1) + HEADER_HEIGHT, BufferedImage.TYPE_3BYTE_BGR);
		Graphics2D graphic = getGraphics2D(image);

		drawHeader(entireAlignment, graphic);

		drawLabelsOnLeft(profileGrid.getSortedAminoAcidSymbols(), profileGrid.getSortedAminoAcidNames(), graphic);

		drawCells(aminoAcidEntries, graphic);

		BufferedImage sideColumn = new BufferedImage(GRID_OFFSET+CELL_WIDTH,(heightInRows * CELL_HEIGHT + 1) + HEADER_HEIGHT, BufferedImage.TYPE_3BYTE_BGR);
		graphic = getGraphics2D(sideColumn);

		drawLabelsOnLeft(profileGrid.getSortedAminoAcidSymbols(), profileGrid.getSortedAminoAcidNames(), graphic);

		try {
			writeImage(image, fileName);
			writeImage(sideColumn, fileName.substring(0, fileName.length()-3) + "sideColumn.png");
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	private void drawHeader(Tier entireAlignment, Graphics2D graphic) {
		Text[] positionRow			= entireAlignment.getPositionInAlignmentRow();
		Text[] majorAminoAcidRow	= entireAlignment.getMajorAminoAcidRow();
		Text[] referenceSequenceRow	= entireAlignment.getReferenceSequenceRow();
		Text[] highlightSequenceRow	= entireAlignment.getHighlightSequenceRow();

		drawRow(positionRow, 1, graphic);
		drawRow(majorAminoAcidRow, 2, graphic);
		drawRow(referenceSequenceRow, 3, graphic);
		drawRow(highlightSequenceRow, 4, graphic);
	}

	private void drawRow(Text[] row, int offset, Graphics2D graphic) {
		graphic.setColor(Color.BLACK);
		if(offset == 1)
			graphic.drawString("Position", 10, offset * CELL_HEIGHT);
		if(offset == 2)
			graphic.drawString("Majority", 10, offset * CELL_HEIGHT);
		if(offset == 3)
			graphic.drawString("Reference", 10, offset * CELL_HEIGHT); // TODO use truncated real sequence name?
		if(offset == 4)
			graphic.drawString("Highlight", 10, offset * CELL_HEIGHT); // TODO toggle label; use truncated name
		for( int i = 0; i < row.length; i++ ) {
			//int y1 = (1 + i)*CELL_HEIGHT-4 + HEADER_HEIGHT;
			graphic.drawString(row[i].getValue(), GRID_OFFSET + CELL_WIDTH * (i+1) + (CELL_WIDTH/2) - (graphic.getFontMetrics().stringWidth(row[i].getValue())/2), offset * CELL_HEIGHT);
		}
	}

	private void drawLabelsOnLeft(String[] symbols, String[] names, Graphics2D graphic) {
		// Default text Color
		graphic.setColor(Color.BLACK);

		for (int i=0; i<symbols.length; i++) {
			int y1 = (1 + i)*CELL_HEIGHT-4 + HEADER_HEIGHT;
			graphic.drawString(symbols[i], 10, y1);

			int y2 = (1 + i)*CELL_HEIGHT-2 + HEADER_HEIGHT;
			graphic.drawString(names[i], 37, y2);
		}
	}

	private void drawCells(Cell[][] cells, Graphics2D graphic) {

		Color currColor = graphic.getColor();
		int xLocation = 1;
		for (int i=0; i<cells.length;i++)
		{
			int yLocation = 1;
			for (int j=0; j<cells[i].length; j++)
			{
				if( cells[i][j].getFormat().getBorderColor() != null)
					drawBorder(graphic, xLocation, yLocation, toColor(cells[i][j].getFormat().getBorderColor()));
				else
					drawBorder(graphic, xLocation, yLocation, Color.LIGHT_GRAY);

				drawCell(graphic, xLocation, yLocation,  cells[i][j]);
				yLocation++;
			}
			xLocation++;
		}
		graphic.setColor(currColor);
	}

	private void drawCell(Graphics2D g, int i, int j, Cell cell) {
		// Format this cell
		Format format = cell.getFormat();

		setBackgroundColor(g, i, j, format);

		setTextAttributes(g, format);

		StringBuffer textToShow = new StringBuffer(4);
		if (cell instanceof Numeric) {
			if( ((Numeric)cell).getValue() != 0.0 )
			{
				if(((Numeric)cell).getValue() < 1.0 && ((Numeric)cell).getValue() != 0.0 )
				{
					DecimalFormat decForm = new DecimalFormat("#.##");
					textToShow.append(decForm.format(((Numeric)cell).getValue()));
				}
				else
					textToShow.append(Math.round(((Numeric)cell).getValue()));

			}
			else
				textToShow.append("");
		} else {
			textToShow.append(cell.toString());
		}

		// Alignment
//		int length = textToShow.length();
//		for (i = length; i < 2; i++ ) {
//			textToShow.insert(0, " ");
//		}
//		System.err.println(textToShow.toString());

		g.drawString(textToShow.toString(), GRID_OFFSET + (i*CELL_WIDTH) + (CELL_WIDTH/2) - (g.getFontMetrics().stringWidth(textToShow.toString())/2), j*CELL_HEIGHT-2+HEADER_HEIGHT);
	}

	private void setBackgroundColor(Graphics2D g, int i, int j,
			Format format) {
		Colour colour = format.getBackgroundColor();
		if(colour == null)
			g.setColor(Color.WHITE);
		else
			g.setColor(toColor(colour));
		g.fillRect(GRID_OFFSET + ((i*CELL_WIDTH))+BORDER_WIDTH, (((j-1)*CELL_HEIGHT)+BORDER_WIDTH+HEADER_HEIGHT), CELL_WIDTH-2*BORDER_WIDTH+2, CELL_HEIGHT-2*BORDER_WIDTH+2);
	}

	private static void setTextAttributes(Graphics2D g, Format format) {
		Colour fontColor = format.getBackgroundColor();
		if( fontColor != null )
		{
			if(fontColor.getDefaultRGB().getGreen() <  255 * .6)
				fontColor = Colour.WHITE;
			else
				fontColor = Colour.BLACK;
		}

		g.setColor(toColor(fontColor));
		if (format.isBold()) {
			Font font = g.getFont();
			Font derivedFont = font.deriveFont(Font.BOLD);
			g.setFont(derivedFont);
		} else {
			Font font = g.getFont();
			Font derivedFont = font.deriveFont(Font.PLAIN);
			g.setFont(derivedFont);
		}
	}

	private void drawBorder(Graphics2D g, int i, int j, Color borderColor) {
		// set color to light gray as default instead of black or "toColor(cell.getFormat().getBorderColor())"
		// Left
		g.setColor(borderColor);
		for( int k = 1; k < BORDER_WIDTH; k++ )
		{
			Rectangle rect = new Rectangle(CELL_WIDTH-2*k+1, CELL_HEIGHT-2*k+1);
			rect.x = GRID_OFFSET + ((i*CELL_WIDTH)) + k;
			rect.y = (((j-1)*CELL_HEIGHT)+HEADER_HEIGHT) + k;
			g.draw(rect);
		}
	}

	private static Color toColor(Colour colour) {
		// Guard Clause
		if (colour == null) return Color.BLACK;
		colour = Colour.getInternalColour(colour.getValue());
		RGB rgb = colour.getDefaultRGB();
		Color color = new Color(rgb.getRed(), rgb.getGreen(), rgb.getBlue() );
		return color;
	}

	private static Graphics2D getGraphics2D(BufferedImage image) {
		Graphics2D g = image.createGraphics();
		// Make  the background white
		// I am sure there is a better way to do this like with the type
		g.setColor(Color.WHITE);
		g.fillRect(0, 0, image.getWidth(), image.getHeight());
		return g;
	}

/** TODO delete these functions?
 * 	private static void print(String[] vals, String message) {
		if (vals != null) {
			System.err.println(message);
			for (int i=0; i<vals.length;i++) {
				System.err.println(vals[i]);
			}
		}
	}
	private static void print(Cell[] vals, String message) {
		if (vals != null) {
			System.err.println(message);
			for (int i=0; i<vals.length;i++) {
				System.err.println(vals[i].toString());
			}
		}
	}

	private static void print(Cell[][] vals, String message) {
		if (vals != null) {
			System.err.println(message);
			for (int i=0; i<vals.length;i++) {
				for (int j=0; j<vals[i].length; j++) {
					System.err.print(vals[i][j].toString());
				}
				System.err.print("\n");
			}
		}
	}
*/

	/**
	 *  Create a BufferedImage from a rectangular region on the screen.
	 *  This will include Swing components JFrame, JDialog and JWindow
	 *  which all extend from Component, not JComponent.
	 *
	 *  @param	 region region on the screen to create image from
	 *  @param	 fileName name of file to be created or null
	 *  @return	image the image for the given region
	 *  @exception AWTException see Robot class constructors
	 *  @exception IOException if an error occurs during writing
	 */
	public BufferedImage createImage(Rectangle region, String fileName)
		throws AWTException, IOException {
		BufferedImage image = new Robot().createScreenCapture( region );
		writeImage(image, fileName);
		return image;
	}

	public static BufferedImage newCreateImage(Component cmp, String filename) {
		Rectangle size = cmp.getBounds(); // need correct size; formerly getSize()
		BufferedImage myImage = new BufferedImage(size.width, size.height, BufferedImage.TYPE_INT_RGB);
		Graphics2D g2 = myImage.createGraphics();
		cmp.paint(g2);
		try {
			OutputStream out = new FileOutputStream(filename);
			//JPEGImageEncoder encoder = JPEGCodec.createJPEGEncoder(out);
			//encoder.encode(myImage);
			ImageIO.write(myImage, "png", out);
			out.close();
		} catch (Exception e) {
		System.out.println(e);
		}
		return null;

	}

	/**
	 *  Write a BufferedImage to a File.
	 *
	 *  @param	 image image to be written
	 *  @param	 fileName name of file to be created
	 *  @exception IOException if an error occurs during writing
	*/
	public void writeImage(BufferedImage image, String fileName)
		throws IOException {
		if (fileName == null) return;

		int offset = fileName.lastIndexOf( "." );

		if (offset == -1) {
			String message = "file suffix was not specified";
			throw new IOException( message );
		}

		String type = fileName.substring(offset + 1);

		if (types.contains(type)) {
			ImageIO.write(image, type, new File( fileName ));
		}
		else {
			String message = "unknown writer file suffix (" + type + ")";
			throw new IOException( message );
		}
	}
}