package jprofilegrid.writers.excel;

import jxl.write.biff.RowsExceededException;
import jxl.write.WriteException;
import jxl.write.WritableSheet;
import jxl.write.WritableCellFormat;
import jxl.write.Number;
import jxl.write.Label;
import jxl.format.Colour;
import jxl.format.Pattern;

public class ExcelDataBlockWriter
{
	private WritableSheet sheet;

	/*
	 * An ExcelDataBlockWriter may only be constructed along with
	 * an associated sheet. This ensures that calls
	 * to the ExcelDataBlockWriter are valid.
	 */
	public ExcelDataBlockWriter(WritableSheet nSheet)
	{
		sheet = nSheet;
	}

	/*
	 * Writes the data with the specified format
	 * at the given location. There are
	 * two such methods: one for String data and one
	 * for numeric data (Double).
	 */
	public void writeString(ExcelTextCell textCell, int col, int row )
	{
		addCell( col, row, textCell );
	}

	public void writeDouble(ExcelNumericCell numericCell, int col, int row )
	{
		addNumericCell( col, row, numericCell );
	}

	/*
	 * Writes the data given in cellData[] with format
	 * cellDataFormats[] at position x, y as the starting
	 * left point in the row. There are
	 * two such methods: one for String data and one
	 * for numeric data (Double).
	 *
	 */
	public void writeRow(ExcelCell[] excelCells, int startCol, int startRow  )
	{
		for( int i = 0, col = startCol; i < excelCells.length; i++, col++ )
			addCell( col, startRow, excelCells[i] );
	}

	/*
	 * Writes the data given in cellData[] with format
	 * cellDataFormats[] at position x, y as the starting
	 * top point in the column. There are
	 * two such methods: one for String data and one
	 * for numeric data (Double).
	 *
	 */
	public void writeCol(ExcelCell[] excelCells, int startCol, int startRow  )
	{
		for( int i = 0, row = startRow; i < excelCells.length; i++, row++ )
			addCell( startCol, row, excelCells[i] );
	}

	 /* Writes the data given in cellData[][] with format
	 * cellDataFormats[][] at position x, y as the starting
	 * upper-left point in the sheet. There are
	 * two such methods: one for String data and one
	 * for numeric data (Double).
	 *
	 */
	public void writeBlock( ExcelCell[][] excelCells, int startCol, int startRow  )
	{
		for( int i = 0, col = startCol; i < excelCells.length; i++, col++ )
			writeCol(excelCells[i], col , startRow);
	}

	/*
	 * These two methods actual perform the addition of the cell to
	 * the sheet at the specified column and row with the given
	 * label and format. Note that this method tries
	 * to add the data as a number (to avoid the annoying 'Convert to
	 * number' warnings from Excel).
	 */
	private void addCell( int col, int row, ExcelCell excelCell )
	{
		if(excelCell instanceof ExcelTextCell)
		{
			ExcelTextCell excelTextCell = (ExcelTextCell)excelCell;

			try
			{
				sheet.addCell(new Number(col, row, Double.parseDouble(excelTextCell.getValue()), excelTextCell.getFormat()));
			}
			catch(NumberFormatException e)
			{
				addTextCellWithoutParsingNumber( col, row, excelTextCell );
			}
			catch(RowsExceededException e)
			{
			}
			catch(WriteException e)
			{
			}
		}
		else
		{
			ExcelNumericCell excelNumericCell = (ExcelNumericCell)excelCell;
			addNumericCell(col, row, excelNumericCell);
		}
	}

	/*
	 * Adds a Text cell without trying to parse a double. Note that this
	 * may lead to warnings from Excel if numeric data is entered this way.
	 */
	private void addTextCellWithoutParsingNumber( int col, int row, ExcelTextCell textCell )
	{
		try
		{
			WritableCellFormat format = textCell.getFormat();

			if( format.getBackgroundColour() == Colour.BLACK )
				format.setBackground(Colour.AUTOMATIC);

			if( format.getBackgroundColour().getValue() == Colour.WHITE.getValue())
				format = extractAllFormattingExceptBackground(format);

			sheet.addCell(new Label(col, row, textCell.getValue(), format));
		}
		catch(RowsExceededException e)
		{
		}
		catch(WriteException e)
		{
		}
	}

	/*
	 * Add a Numeric Cell. Note that the color-correction for white cells is applied.
	 */
	private void addNumericCell( int col, int row, ExcelNumericCell numericCell)
	{
		try
		{
			WritableCellFormat format = numericCell.getFormat();
			if( format.getBackgroundColour() == Colour.BLACK )
				format.setBackground(Colour.AUTOMATIC);

			if( format.getBackgroundColour().getValue() == Colour.WHITE.getValue())
				format = extractAllFormattingExceptBackground(format);

			if( numericCell.getValue() != 0)
				sheet.addCell(new Number(col, row, numericCell.getValue(), format));
			else
				sheet.addCell(new Label(col, row, "", format));
		}
		catch(RowsExceededException e)
		{
		}
		catch(WriteException e)
		{
		}
	}

	private WritableCellFormat extractAllFormattingExceptBackground(WritableCellFormat oldFormat)
	{
		try
		{
			oldFormat.setBackground(Colour.WHITE, Pattern.NONE);

			return oldFormat;
		}
		catch(WriteException e)
		{
		}

		return null;
	}
}
